package riakdemo.backend;

import java.net.UnknownHostException;
import java.util.List;
import java.util.concurrent.ExecutionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.basho.riak.client.api.RiakClient;
import com.basho.riak.client.api.cap.VClock;
import com.basho.riak.client.api.commands.kv.DeleteValue;
import com.basho.riak.client.api.commands.kv.FetchValue;
import com.basho.riak.client.api.commands.kv.FetchValue.Option;
import com.basho.riak.client.api.commands.kv.StoreValue;
import com.basho.riak.client.api.commands.kv.StoreValue.Builder;
import com.basho.riak.client.core.query.Location;
import com.basho.riak.client.core.query.Namespace;

import riakdemo.tools.Converter;
import riakdemo.tools.Pair;

public class BackendSession {

	private static final Logger LOGGER = LoggerFactory.getLogger(BackendSession.class);

	private static RiakClient client;

	public static RiakClient getClient() throws UnknownHostException {
		return client;
	}

	public BackendSession(String contactPoint, int contactPointPort) throws BackendException {
		try {
			client = RiakClient.newClient(contactPointPort, contactPoint);
		} catch (UnknownHostException e) {
			throw new BackendException("Could not connect to the cluster. " + e.getMessage() + ".", e);
		}
	}

	public void upsertUser(Namespace namespace, String key, User user, VClock vclock, boolean useVClock) throws BackendException {
		Location location = new Location(namespace, key);

		Builder builder = new StoreValue.Builder(user).withLocation(location);

		if (useVClock)
			builder = builder.withVectorClock(vclock);

		StoreValue storeOp = builder.build();

		try {
			client.execute(storeOp);
		} catch (ExecutionException e) {
			throw new BackendException("Could not perform an upsert operation. " + e.getMessage() + ".", e);
		} catch (InterruptedException e) {
			throw new BackendException("Could not perform an upsert operation. " + e.getMessage() + ".", e);
		}

		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append("Entry " + location.toString() + " upserted ");
		if (useVClock)
			stringBuilder.append("with vclock " + (vclock == null ? "null" : Converter.toHex(vclock.getBytes())));
		else
			stringBuilder.append("with no vclock");
		LOGGER.info(stringBuilder.toString());
	}
	
	public Pair<List<User>, VClock> fetchUserSiblings(Namespace namespace, String key) throws BackendException {
		Location location = new Location(namespace, key);
		FetchValue fetchOp = new FetchValue.Builder(location).withOption(Option.DELETED_VCLOCK, true).build();
		List<User> siblings = null;
		VClock vclock = null;
		try {
			FetchValue.Response response = client.execute(fetchOp);
			siblings = response.getValues(User.class);
			vclock = response.getVectorClock();
		} catch (ExecutionException e) {
			throw new BackendException("Could not perform a fetch operation. " + e.getMessage() + ".", e);
		} catch (InterruptedException e) {
			throw new BackendException("Could not perform a fetch operation.  " + e.getMessage() + ".", e);
		}

		LOGGER.info("Entry " + location.toString() + " fetched with vclock "
				+ (vclock == null ? "null" : Converter.toHex(vclock.getBytes())));

		return new Pair<List<User>, VClock>(siblings, vclock);
	}

	public void deleteUser(Namespace namespace, String key, VClock vclock, boolean useVClock) throws BackendException {
		Location location = new Location(namespace, key);
		DeleteValue.Builder builder = new DeleteValue.Builder(location);

		if (useVClock)
			builder = builder.withVClock(vclock);

		DeleteValue deleteOp = builder.build();

		try {
			client.execute(deleteOp);
		} catch (ExecutionException e) {
			throw new BackendException("Could not perform a delete operation. " + e.getMessage() + ".", e);
		} catch (InterruptedException e) {
			throw new BackendException("Could not perform a delete operation. " + e.getMessage() + ".", e);
		}

		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append("Entry " + location.toString() + " deleted ");
		if (useVClock)
			stringBuilder.append("with vclock " + (vclock == null ? "null" : Converter.toHex(vclock.getBytes())));
		else
			stringBuilder.append("with no vclock");
		LOGGER.info(stringBuilder.toString());
	}
	
	protected void finalize() {
		try {
			if (client != null) {
				client.shutdown();
			}
		} catch (Exception e) {
			LOGGER.error("Could not close existing cluster", e);
		}
	}

}
