/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.formatter;

import java.io.IOException;
import java.io.StringReader;
import java.util.Arrays;
import java.util.Comparator;

import org.eclipse.jdt.core.compiler.CharOperation;
import org.eclipse.jdt.core.compiler.InvalidInputException;
import org.eclipse.jdt.core.formatter.CodeFormatter;
import org.eclipse.jdt.internal.compiler.ASTVisitor;
import org.eclipse.jdt.internal.compiler.ast.Annotation;
import org.eclipse.jdt.internal.compiler.lookup.BlockScope;
import org.eclipse.jdt.internal.compiler.parser.Scanner;
import org.eclipse.jdt.internal.compiler.parser.ScannerHelper;
import org.eclipse.jdt.internal.compiler.parser.TerminalTokens;
import org.eclipse.jdt.internal.compiler.util.Util;
import org.eclipse.jdt.internal.core.util.CodeSnippetParsingUtil;
import org.eclipse.jdt.internal.core.util.RecordedParsingInformation;
import org.eclipse.jdt.internal.formatter.align.Alignment;
import org.eclipse.jdt.internal.formatter.align.AlignmentException;
import org.eclipse.jdt.internal.formatter.comment.CommentFormatterUtil;
import org.eclipse.jdt.internal.formatter.comment.HTMLEntity2JavaReader;
import org.eclipse.jdt.internal.formatter.comment.IJavaDocTagConstants;
import org.eclipse.jdt.internal.formatter.comment.Java2HTMLEntityReader;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DefaultLineTracker;
import org.eclipse.jface.text.ILineTracker;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.Region;
import org.eclipse.text.edits.MalformedTreeException;
import org.eclipse.text.edits.MultiTextEdit;
import org.eclipse.text.edits.ReplaceEdit;
import org.eclipse.text.edits.TextEdit;

/**
 * This class is responsible for dumping formatted source
 * @since 2.1
 */
public class Scribe implements IJavaDocTagConstants {

	private static final int INITIAL_SIZE = 100;
	
	private boolean checkLineWrapping;
	/** one-based column */
	public int column;
	private int[][] commentPositions;
		
	// Most specific alignment. 
	public Alignment currentAlignment;
	public int currentToken;
	
	// edits management
	private OptimizedReplaceEdit[] edits;
	public int editsIndex;
	
	public CodeFormatterVisitor formatter;
	public int indentationLevel;	
	public int lastNumberOfNewLines;
	public int line;
	
	private int[] lineEnds;
	private int maxLines;
	private String lineSeparator;
	public Alignment memberAlignment;
	public boolean needSpace = false;
	
	public int nlsTagCounter;
	public int pageWidth;
	public boolean pendingSpace = false;

	public Scanner scanner;
	public int scannerEndPosition;
	public int tabLength;	
	public int indentationSize;
	private IRegion[] regions;
	private IRegion[] adaptedRegions;
	public int tabChar;
	public int numberOfIndentations;
	private boolean useTabsOnlyForLeadingIndents;

	/** indent empty lines*/
	private final boolean indentEmptyLines;
	
//	boolean formatJavadocComment;
//	boolean formatBlockComment;
//	boolean formatLineComment;
//	boolean includeComments;
	private int formatComments = 0; // false
	private static final int INCLUDE_BLOCK_COMMENTS = CodeFormatter.F_INCLUDE_COMMENTS | CodeFormatter.K_MULTI_LINE_COMMENT;
	private static final int INCLUDE_JAVA_DOC = CodeFormatter.F_INCLUDE_COMMENTS | CodeFormatter.K_JAVA_DOC;
	private static final int INCLUDE_LINE_COMMENTS = CodeFormatter.F_INCLUDE_COMMENTS | CodeFormatter.K_SINGLE_LINE_COMMENT;
	private int headerEndPosition = -1;
	
	// New way to format javadoc
	private FormatterCommentParser formatterCommentParser; // specialized parser to format comments

	Scribe(CodeFormatterVisitor formatter, long sourceLevel, IRegion[] regions, CodeSnippetParsingUtil codeSnippetParsingUtil, boolean includeComments) {
		this.scanner = new Scanner(true, true, false/*nls*/, sourceLevel/*sourceLevel*/, null/*taskTags*/, null/*taskPriorities*/, true/*taskCaseSensitive*/);
		this.formatter = formatter;
		this.pageWidth = formatter.preferences.page_width;
		this.tabLength = formatter.preferences.tab_size;
		this.indentationLevel= 0; // initialize properly
		this.numberOfIndentations = 0;
		this.useTabsOnlyForLeadingIndents = formatter.preferences.use_tabs_only_for_leading_indentations;
		this.indentEmptyLines = formatter.preferences.indent_empty_lines;
		this.tabChar = formatter.preferences.tab_char;
		if (this.tabChar == DefaultCodeFormatterOptions.MIXED) {
			this.indentationSize = formatter.preferences.indentation_size;
		} else {
			this.indentationSize = this.tabLength;
		}
		this.lineSeparator = formatter.preferences.line_separator;
		this.indentationLevel = formatter.preferences.initial_indentation_level * this.indentationSize;
		this.regions= regions;
		if (codeSnippetParsingUtil != null) {
			final RecordedParsingInformation information = codeSnippetParsingUtil.recordedParsingInformation;
			if (information != null) {
				this.lineEnds = information.lineEnds;
				this.commentPositions = information.commentPositions;
			}
		}
		if (formatter.preferences.comment_format_line_comment) this.formatComments |= CodeFormatter.K_SINGLE_LINE_COMMENT;
		if (formatter.preferences.comment_format_block_comment) this.formatComments |= CodeFormatter.K_MULTI_LINE_COMMENT;
		if (formatter.preferences.comment_format_javadoc_comment) this.formatComments |= CodeFormatter.K_JAVA_DOC;
		if (includeComments) this.formatComments |= CodeFormatter.F_INCLUDE_COMMENTS;
		reset();
	}
	
	/**
	 * This method will adapt the selected regions if needed.
	 * If a region should be adapted (see isAdaptableRegion(IRegion))
	 * retrieve correct upper and lower bounds and replace the region.
	 */
	private void adaptRegions() {
		this.adaptedRegions = new IRegion[this.regions.length];
		for (int i = 0, max = this.regions.length; i < max; i++) {
			IRegion aRegion = this.regions[i];
			int offset = aRegion.getOffset();
			if (offset > 0) {
				int length = aRegion.getLength();
				if (isAdaptableRegion(offset, length)) {
					// if we have a selection, search for overlapping edits
					int upperBound = offset;
					int lowerBound = 0;
					boolean upperFound = false;
					int regionEnd = offset + length;
					for (int j = 0, max2 = this.editsIndex - 1; j <= max2; j++) {
						// search for lower bound
						int editOffset = this.edits[j].offset;
						if (upperFound && lowerBound == 0) {
							int editLength = this.edits[j].length;
							if (editOffset == regionEnd) { // matching edit found
								lowerBound = regionEnd;
								break;
							} else if (editOffset + editLength < regionEnd) {
								continue;
							} else {
								lowerBound = editOffset + editLength; // upper and lower bounds found
								break;
							}
							// search for upper bound
						} else {
							int next = j+1;
							if (next == max2) {
								// https://bugs.eclipse.org/bugs/show_bug.cgi?id=213284
								// checked all edits, no upper bound found: leave the loop
								break;
							}
							if (this.edits[next].offset < offset) {
								continue;
							} else {
								upperBound = editOffset;
								upperFound = true;
								// verify if region end is at EOF
								if (this.scannerEndPosition == regionEnd) {
									lowerBound = this.scannerEndPosition - 1;
									break;
								}
							}
						}
					}
					if (lowerBound != 0) {
						if (offset != upperBound || regionEnd != lowerBound) { // ensure we found a different region
							this.adaptedRegions[i] = new Region(upperBound,
									lowerBound - upperBound);
						}
						// keep other unadaptable region
					} else {
						this.adaptedRegions[i] = this.regions[i];
					}
				} else {
					this.adaptedRegions[i] = this.regions[i];
				}
			} else {
				this.adaptedRegions[i] = this.regions[i];
			}
		}
	}

	private final void addDeleteEdit(int start, int end) {
		if (this.edits.length == this.editsIndex) {
			// resize
			resize();
		}
		addOptimizedReplaceEdit(start, end - start + 1, Util.EMPTY_STRING);
	}

	public final void addInsertEdit(int insertPosition, String insertedString) {
		if (this.edits.length == this.editsIndex) {
			// resize
			resize();
		}
		addOptimizedReplaceEdit(insertPosition, 0, insertedString);
	}

	private final void addOptimizedReplaceEdit(int offset, int length, String replacement) {
		if (this.editsIndex > 0) {
			// try to merge last two edits
			final OptimizedReplaceEdit previous = this.edits[this.editsIndex-1];
			final int previousOffset = previous.offset;
			final int previousLength = previous.length;
			final int endOffsetOfPreviousEdit = previousOffset + previousLength;
			final int replacementLength = replacement.length();
			final String previousReplacement = previous.replacement;
			final int previousReplacementLength = previousReplacement.length();
			if (previousOffset == offset && previousLength == length && (replacementLength == 0 || previousReplacementLength == 0)) {
				if (this.currentAlignment != null) {
					final Location location = this.currentAlignment.location;
					if (location.editsIndex == this.editsIndex) {
						location.editsIndex--;
						location.textEdit = previous;
					}
				}
				this.editsIndex--;
				return;
			}
			if (endOffsetOfPreviousEdit == offset) {
				if (length != 0) {
					if (replacementLength != 0) {
						this.edits[this.editsIndex - 1] = new OptimizedReplaceEdit(previousOffset, previousLength + length, previousReplacement + replacement);
					} else if (previousLength + length == previousReplacementLength) {
						// check the characters. If they are identical, we can get rid of the previous edit
						boolean canBeRemoved = true;
						loop: for (int i = previousOffset; i < previousOffset + previousReplacementLength; i++) {
							if (scanner.source[i] != previousReplacement.charAt(i - previousOffset)) {
								this.edits[this.editsIndex - 1] = new OptimizedReplaceEdit(previousOffset, previousReplacementLength, previousReplacement);
								canBeRemoved = false;
								break loop;
							}
						}
						if (canBeRemoved) {
							if (this.currentAlignment != null) {
								final Location location = this.currentAlignment.location;
								if (location.editsIndex == this.editsIndex) {
									location.editsIndex--;
									location.textEdit = previous;
								}
							}
							this.editsIndex--;
						}
					} else {
						this.edits[this.editsIndex - 1] = new OptimizedReplaceEdit(previousOffset, previousLength + length, previousReplacement);
					}
				} else {
					if (replacementLength != 0) {
						this.edits[this.editsIndex - 1] = new OptimizedReplaceEdit(previousOffset, previousLength, previousReplacement + replacement);
					}
				}
			} else if ((offset + length == previousOffset) && (previousLength + length == replacementLength + previousReplacementLength)) {
				// check if both edits corresponds to the orignal source code
				boolean canBeRemoved = true;
				String totalReplacement = replacement + previousReplacement;
				loop: for (int i = 0; i < previousLength + length; i++) {
					if (scanner.source[i + offset] != totalReplacement.charAt(i)) {
						this.edits[this.editsIndex - 1] = new OptimizedReplaceEdit(offset, previousLength + length, totalReplacement);
						canBeRemoved = false;
						break loop;
					}
				}
				if (canBeRemoved) {
					if (this.currentAlignment != null) {
						final Location location = this.currentAlignment.location;
						if (location.editsIndex == this.editsIndex) {
							location.editsIndex--;
							location.textEdit = previous;
						}
					}
					this.editsIndex--;
				}
			} else {
				this.edits[this.editsIndex++] = new OptimizedReplaceEdit(offset, length, replacement);
			}
		} else {
			this.edits[this.editsIndex++] = new OptimizedReplaceEdit(offset, length, replacement);
		}
	}
	
	public final void addReplaceEdit(int start, int end, String replacement) {
		if (this.edits.length == this.editsIndex) {
			// resize
			resize();
		}
		addOptimizedReplaceEdit(start,  end - start + 1, replacement);
	}

	public void alignFragment(Alignment alignment, int fragmentIndex){
		alignment.fragmentIndex = fragmentIndex;
		alignment.checkColumn();
		alignment.performFragmentEffect();
	}
	
	public void checkNLSTag(int sourceStart) {
		if (hasNLSTag(sourceStart)) {
			this.nlsTagCounter++;
		}
	}

	public Alignment createAlignment(String name, int mode, int count, int sourceRestart){
		return createAlignment(name, mode, Alignment.R_INNERMOST, count, sourceRestart);
	}

	public Alignment createAlignment(String name, int mode, int count, int sourceRestart, boolean adjust){
		return createAlignment(name, mode, Alignment.R_INNERMOST, count, sourceRestart, adjust);
	}
	
	public Alignment createAlignment(String name, int mode, int tieBreakRule, int count, int sourceRestart){
		return createAlignment(name, mode, tieBreakRule, count, sourceRestart, this.formatter.preferences.continuation_indentation, false);
	}

	public Alignment createAlignment(String name, int mode, int count, int sourceRestart, int continuationIndent, boolean adjust){
		return createAlignment(name, mode, Alignment.R_INNERMOST, count, sourceRestart, continuationIndent, adjust);
	}

	public Alignment createAlignment(String name, int mode, int tieBreakRule, int count, int sourceRestart, int continuationIndent, boolean adjust){
		Alignment alignment = new Alignment(name, mode, tieBreakRule, this, count, sourceRestart, continuationIndent);
		// adjust break indentation
		if (adjust && this.memberAlignment != null) {
			Alignment current = this.memberAlignment;
			while (current.enclosing != null) {
				current = current.enclosing;
			}
			if ((current.mode & Alignment.M_MULTICOLUMN) != 0) {
				final int indentSize = this.indentationSize;
				switch(current.chunkKind) {
					case Alignment.CHUNK_METHOD :
					case Alignment.CHUNK_TYPE :
						if ((mode & Alignment.M_INDENT_BY_ONE) != 0) {
							alignment.breakIndentationLevel = this.indentationLevel + indentSize;
						} else {
							alignment.breakIndentationLevel = this.indentationLevel + continuationIndent * indentSize;
						}
						alignment.update();
						break;
					case Alignment.CHUNK_FIELD :
						if ((mode & Alignment.M_INDENT_BY_ONE) != 0) {
							alignment.breakIndentationLevel = current.originalIndentationLevel + indentSize;
						} else {
							alignment.breakIndentationLevel = current.originalIndentationLevel + continuationIndent * indentSize;
						}
						alignment.update();
						break;
				}
			} else {
				switch(current.mode & Alignment.SPLIT_MASK) {
					case Alignment.M_COMPACT_SPLIT :
					case Alignment.M_COMPACT_FIRST_BREAK_SPLIT :
					case Alignment.M_NEXT_PER_LINE_SPLIT :
					case Alignment.M_NEXT_SHIFTED_SPLIT :
					case Alignment.M_ONE_PER_LINE_SPLIT :
						final int indentSize = this.indentationSize;
						switch(current.chunkKind) {
							case Alignment.CHUNK_METHOD :
							case Alignment.CHUNK_TYPE :
								if ((mode & Alignment.M_INDENT_BY_ONE) != 0) {
									alignment.breakIndentationLevel = this.indentationLevel + indentSize;
								} else {
									alignment.breakIndentationLevel = this.indentationLevel + continuationIndent * indentSize;
								}
								alignment.update();
								break;
							case Alignment.CHUNK_FIELD :
								if ((mode & Alignment.M_INDENT_BY_ONE) != 0) {
									alignment.breakIndentationLevel = current.originalIndentationLevel + indentSize;
								} else {
									alignment.breakIndentationLevel = current.originalIndentationLevel + continuationIndent * indentSize;
								}
								alignment.update();
								break;
						}
						break;
				}
			}
		}
		return alignment; 
	}

	public Alignment createMemberAlignment(String name, int mode, int count, int sourceRestart) {
		Alignment mAlignment = createAlignment(name, mode, Alignment.R_INNERMOST, count, sourceRestart);
		mAlignment.breakIndentationLevel = this.indentationLevel;
		return mAlignment;
	}
	
	public void enterAlignment(Alignment alignment){
		alignment.enclosing = this.currentAlignment;
		alignment.location.lastLocalDeclarationSourceStart = this.formatter.lastLocalDeclarationSourceStart;
		this.currentAlignment = alignment;
	}

	public void enterMemberAlignment(Alignment alignment) {
		alignment.enclosing = this.memberAlignment;
		alignment.location.lastLocalDeclarationSourceStart = this.formatter.lastLocalDeclarationSourceStart;
		this.memberAlignment = alignment;
	}

	public void exitAlignment(Alignment alignment, boolean discardAlignment){
		Alignment current = this.currentAlignment;
		while (current != null){
			if (current == alignment) break;
			current = current.enclosing;
		}
		if (current == null) {
			throw new AbortFormatting("could not find matching alignment: "+alignment); //$NON-NLS-1$
		}
		this.indentationLevel = alignment.location.outputIndentationLevel;
		this.numberOfIndentations = alignment.location.numberOfIndentations;
		this.formatter.lastLocalDeclarationSourceStart = alignment.location.lastLocalDeclarationSourceStart;	
		if (discardAlignment){ 
			this.currentAlignment = alignment.enclosing;
		}
	}
	
	public void exitMemberAlignment(Alignment alignment){
		Alignment current = this.memberAlignment;
		while (current != null){
			if (current == alignment) break;
			current = current.enclosing;
		}
		if (current == null) {
			throw new AbortFormatting("could not find matching alignment: "+alignment); //$NON-NLS-1$
		}
		this.indentationLevel = current.location.outputIndentationLevel;
		this.numberOfIndentations = current.location.numberOfIndentations;
		this.formatter.lastLocalDeclarationSourceStart = alignment.location.lastLocalDeclarationSourceStart;	
		this.memberAlignment = current.enclosing;
	}
	
	public Alignment getAlignment(String name){
		if (this.currentAlignment != null) {
			return this.currentAlignment.getAlignment(name);
		}
		return null;
	}
	
	/** 
	 * Answer actual indentation level based on true column position
	 * @return int
	 */
	public int getColumnIndentationLevel() {
		return this.column - 1;
	}	
	
	public final int getCommentIndex(int position) {
		if (this.commentPositions == null)
			return -1;
		int length = this.commentPositions.length;
		if (length == 0) {
			return -1;
		}
		int g = 0, d = length - 1;
		int m = 0;
		while (g <= d) {
			m = g + (d - g) / 2;
			int bound = this.commentPositions[m][1];
			if (bound < 0) {
				bound = -bound;
			}
			if (bound < position) {
				g = m + 1;
			} else if (bound > position) {
				d = m - 1;
			} else {
				return m;
			}
		}
		return -(g + 1);
	}

	private IRegion getCoveringRegion(int offset, int end) {
		int index = getIndexOfRegionAt(offset);

		if (index < 0) {
			index = -(index + 1);
			index--;
			if (index < 0) {
				return null;
			}
		}

		IRegion region = this.adaptedRegions[index];
		if ((region.getOffset() <= offset) && (end <= region.getOffset() + region.getLength() - 1)) {
			return region;
		}
		return null;
	}
	
	private int getCurrentCommentOffset(int start) {
		int linePtr = -Arrays.binarySearch(this.lineEnds, start);
		int offset = 0;
		int beginningOfLine = this.getLineEnd(linePtr - 1);
		if (beginningOfLine == -1) {
			beginningOfLine = 0;
		}
		int currentStartPosition = start;
		char[] source = scanner.source;

		// find the position of the beginning of the line containing the comment
		while (beginningOfLine > currentStartPosition) {
			if (linePtr > 0) {
				beginningOfLine = this.getLineEnd(--linePtr);
			} else {
				beginningOfLine = 0;
				break;
			}
		}
		for (int i = currentStartPosition - 1; i >= beginningOfLine ; i--) {
			char currentCharacter = source[i];
			switch (currentCharacter) {
				case '\t' :
					offset += this.tabLength;
					break;
				case ' ' :
					offset++;
					break;
				case '\r' :
				case '\n' :
					break;
				default:
					return offset;
			}
		}
		return offset;
	}

	public String getEmptyLines(int linesNumber) {
		if (this.nlsTagCounter > 0) {
			return Util.EMPTY_STRING;
		}
		StringBuffer buffer = new StringBuffer();
		if (lastNumberOfNewLines == 0) {
			linesNumber++; // add an extra line breaks
			for (int i = 0; i < linesNumber; i++) {
				if (indentEmptyLines) printIndentationIfNecessary(buffer);
				buffer.append(this.lineSeparator);
			}
			lastNumberOfNewLines += linesNumber;
			line += linesNumber;
			column = 1;
			needSpace = false;
			this.pendingSpace = false;
		} else if (lastNumberOfNewLines == 1) {
			for (int i = 0; i < linesNumber; i++) {
				if (indentEmptyLines) printIndentationIfNecessary(buffer);
				buffer.append(this.lineSeparator);
			}
			lastNumberOfNewLines += linesNumber;
			line += linesNumber;
			column = 1;
			needSpace = false;
			this.pendingSpace = false;
		} else {
			if ((lastNumberOfNewLines - 1) >= linesNumber) {
				// there is no need to add new lines
				return Util.EMPTY_STRING;
			}
			final int realNewLineNumber = linesNumber - lastNumberOfNewLines + 1;
			for (int i = 0; i < realNewLineNumber; i++) {
				if (indentEmptyLines) printIndentationIfNecessary(buffer);
				buffer.append(this.lineSeparator);
			}
			lastNumberOfNewLines += realNewLineNumber;
			line += realNewLineNumber;
			column = 1;
			needSpace = false;
			this.pendingSpace = false;
		}
		return String.valueOf(buffer);
	}

	private int getIndexOfRegionAt(int offset) {
		if (this.adaptedRegions.length == 1) {
			int offset2 = this.adaptedRegions[0].getOffset();
			if (offset2 == offset) {
				return 0;
			}
			return offset2 < offset ? -2 : -1; 
		}
		return Arrays.binarySearch(this.adaptedRegions, new Region(offset, 0), new Comparator() {
			public int compare(Object o1, Object o2) {
				int r1Offset = ((IRegion)o1).getOffset();
				int r2Offset = ((IRegion)o2).getOffset();
				
				return r1Offset - r2Offset;
			}
		});
	}

	public OptimizedReplaceEdit getLastEdit() {
		if (this.editsIndex > 0) {
			return this.edits[this.editsIndex - 1];
		}
		return null;
	}
	
	public final int getLineEnd(int lineNumber) {
		if (this.lineEnds == null) 
			return -1;
		if (lineNumber >= this.lineEnds.length + 1) 
			return this.scannerEndPosition;
		if (lineNumber <= 0) 
			return -1;
		return this.lineEnds[lineNumber-1]; // next line start one character behind the lineEnd of the previous line	
	}
	
	Alignment getMemberAlignment() {
		return this.memberAlignment;
	}
	
	public String getNewLine() {
		if (this.nlsTagCounter > 0) {
			return Util.EMPTY_STRING;
		}
		if (lastNumberOfNewLines >= 1) {
			column = 1; // ensure that the scribe is at the beginning of a new line
			return Util.EMPTY_STRING;
		}
		line++;
		lastNumberOfNewLines = 1;
		column = 1;
		needSpace = false;
		this.pendingSpace = false;
		return this.lineSeparator;
	}

	/** 
	 * Answer next indentation level based on column estimated position
	 * (if column is not indented, then use indentationLevel)
	 */
	public int getNextIndentationLevel(int someColumn) {
		int indent = someColumn - 1;
		if (indent == 0)
			return this.indentationLevel;
		if (this.tabChar == DefaultCodeFormatterOptions.TAB) {
			if (this.useTabsOnlyForLeadingIndents) {
				return indent;
			}
			int rem = indent % this.indentationSize;
			int addition = rem == 0 ? 0 : this.indentationSize - rem; // round to superior
			return indent + addition;
		}
		return indent;
	}

	private String getPreserveEmptyLines(int count) {
		if (count > 0) {
			if (this.formatter.preferences.number_of_empty_lines_to_preserve != 0) {
				int linesToPreserve = Math.min(count, this.formatter.preferences.number_of_empty_lines_to_preserve);
				return this.getEmptyLines(linesToPreserve);
			}
			return getNewLine();
		}
		return Util.EMPTY_STRING;
	}

	private IRegion getRegionAt(int offset) {
		int index = getIndexOfRegionAt(offset);
		if (index < 0) {
			return null;
		}
		
		return this.regions[index];
	}
	
	public TextEdit getRootEdit() {
		// https://bugs.eclipse.org/bugs/show_bug.cgi?id=208541
		adaptRegions();
		
		MultiTextEdit edit = null;
		int regionsLength = this.adaptedRegions.length;
		int textRegionStart;
		int textRegionEnd;
		if (regionsLength == 1) {
			IRegion lastRegion = this.adaptedRegions[0];
			textRegionStart = lastRegion.getOffset();
			textRegionEnd = textRegionStart + lastRegion.getLength();
		} else {
			textRegionStart = this.adaptedRegions[0].getOffset();
			IRegion lastRegion = this.adaptedRegions[regionsLength - 1];
			textRegionEnd = lastRegion.getOffset() + lastRegion.getLength();
		}
		
		int length = textRegionEnd - textRegionStart + 1;
		if (textRegionStart <= 0) {
			if (length <= 0) {
				edit = new MultiTextEdit(0, 0);
			} else {
				edit = new MultiTextEdit(0, textRegionEnd);
			}
		} else {
			edit = new MultiTextEdit(textRegionStart, length - 1);
		}
		for (int i= 0, max = this.editsIndex; i < max; i++) {
			OptimizedReplaceEdit currentEdit = edits[i];
			if (isValidEdit(currentEdit)) {
				try {
					edit.addChild(new ReplaceEdit(currentEdit.offset, currentEdit.length, currentEdit.replacement));
				}
				catch (MalformedTreeException ex) {
					// log exception in case of error
					CommentFormatterUtil.log(ex);
 					throw ex;
				}
			}
		}
		this.edits = null;
		return edit;
	}
	
	public void handleLineTooLong() {
		// search for closest breakable alignment, using tiebreak rules
		// look for outermost breakable one
		int relativeDepth = 0, outerMostDepth = -1;
		Alignment targetAlignment = this.currentAlignment;
		while (targetAlignment != null){
			if (targetAlignment.tieBreakRule == Alignment.R_OUTERMOST && targetAlignment.couldBreak()){
				outerMostDepth = relativeDepth;
			}
			targetAlignment = targetAlignment.enclosing;
			relativeDepth++;
		}
		if (outerMostDepth >= 0) {
			throw new AlignmentException(AlignmentException.LINE_TOO_LONG, outerMostDepth);
		}
		// look for innermost breakable one
		relativeDepth = 0;
		targetAlignment = this.currentAlignment;
		while (targetAlignment != null){
			if (targetAlignment.couldBreak()){
				throw new AlignmentException(AlignmentException.LINE_TOO_LONG, relativeDepth);
			}
			targetAlignment = targetAlignment.enclosing;
			relativeDepth++;
		}
		// did not find any breakable location - proceed
	}

	/*
	 * Check if there is a NLS tag on this line. If yes, return true, returns false otherwise.
	 */
	private boolean hasNLSTag(int sourceStart) {
		// search the last comment where commentEnd < current lineEnd
		if (this.lineEnds == null) return false;
		int index = Arrays.binarySearch(this.lineEnds, sourceStart);
		int currentLineEnd = this.getLineEnd(-index);
		if (currentLineEnd != -1) {
			int commentIndex = getCommentIndex(currentLineEnd);
			if (commentIndex < 0) {
				commentIndex = -commentIndex - 2;
			}
			if (commentIndex >= 0 && commentIndex < this.commentPositions.length) {
				int start = this.commentPositions[commentIndex][0];
				if (start < 0) {
					start = -start;
					// check that we are on the same line
					int lineIndexForComment = Arrays.binarySearch(this.lineEnds, start);
					if (lineIndexForComment == index) {
						return CharOperation.indexOf(Scanner.TAG_PREFIX, this.scanner.source, true, start, currentLineEnd) != -1;
					}
				}
			}
		}
		return false;
	}

	private boolean includesBlockComments() {
	    return ((this.formatComments & INCLUDE_BLOCK_COMMENTS) == INCLUDE_BLOCK_COMMENTS && this.headerEndPosition < this.scanner.currentPosition) ||
	    	(this.formatter.preferences.comment_format_header && this.headerEndPosition >= this.scanner.currentPosition);
    }

	private boolean includesJavadocComments() {
	    return ((this.formatComments & INCLUDE_JAVA_DOC) == INCLUDE_JAVA_DOC && this.headerEndPosition < this.scanner.currentPosition) ||
	    	(this.formatter.preferences.comment_format_header && this.headerEndPosition >= this.scanner.currentPosition);
    }

	private boolean includesLineComments() {
	    return ((this.formatComments & INCLUDE_LINE_COMMENTS) == INCLUDE_LINE_COMMENTS && this.headerEndPosition < this.scanner.currentPosition) ||
	    	(this.formatter.preferences.comment_format_header && this.headerEndPosition >= this.scanner.currentPosition);
    }

	boolean includesComments() {
	    return (this.formatComments & CodeFormatter.F_INCLUDE_COMMENTS) != 0;
    }

	public void indent() {
		this.indentationLevel += this.indentationSize;
		this.numberOfIndentations++;
	}	

	/**
	 * @param compilationUnitSource
	 */
	public void initializeScanner(char[] compilationUnitSource) {
		this.scanner.setSource(compilationUnitSource);
		this.scannerEndPosition = compilationUnitSource.length;
		this.scanner.resetTo(0, this.scannerEndPosition - 1);
		this.edits = new OptimizedReplaceEdit[INITIAL_SIZE];
		this.maxLines = this.lineEnds == null ? -1 : this.lineEnds.length - 1;
		this.scanner.lineEnds = this.lineEnds;
		this.scanner.linePtr = this.maxLines;
		// Comment following line to de-activate new Javadoc formatter
		initFormatterCommentParser();
	}
	
	private void initFormatterCommentParser() {
		if (this.formatterCommentParser == null) {
			this.formatterCommentParser = new FormatterCommentParser(null);
		}
		this.formatterCommentParser.scanner.setSource(this.scanner.source);
		this.formatterCommentParser.source = this.scanner.source;
		this.formatterCommentParser.scanner.lineEnds = this.lineEnds;
		this.formatterCommentParser.scanner.linePtr = this.maxLines;
	}
	
	/**
	 * Returns whether the given region should be adpated of not.
	 * A region should be adapted only if:
	 * - region does not exceed the page width
	 * - on a single line when more than one line in CU
	 * @param offset the offset of the region to consider
	 * @param length the length of the region to consider
	 * @return boolean true if line should be adapted, false otherwhise
	 */
	private boolean isAdaptableRegion(int offset, int length) {
		int regionEnd = offset + length;
		
		// first check region width
		if (regionEnd > this.pageWidth) {
			return false;
		}
		
		int numberOfLineEnds = this.lineEnds != null && this.lineEnds.length > 0 ? this.lineEnds.length : 0;
		if (this.line == numberOfLineEnds + 1) {
			// https://bugs.eclipse.org/bugs/show_bug.cgi?id=213283
			return true; // last line of the CU
		}
		
		if (this.line > 1 && numberOfLineEnds > 0) { // CU has more than one line
			// https://bugs.eclipse.org/bugs/show_bug.cgi?id=222182
			// take the max number of line ends as right bound when searching for line number
			int lineNumber = Util.getLineNumber(offset, this.lineEnds, 0, numberOfLineEnds);
			int lineEnd = this.getLineEnd(lineNumber);
			if (regionEnd > lineEnd) {
				// if more than one line selected, check whether selection is at line end
				for (int i = lineNumber + 1 ; i <=  numberOfLineEnds ; i++) {
					int nextLineEnd = this.getLineEnd(i);
					// accept both line ends and line starts
					if (regionEnd == nextLineEnd) {
						return length > 1; // except when formatting a single character
					} else if (regionEnd == lineEnd + 1 || regionEnd == nextLineEnd + 1) {
						return true;
					}
				}
				return false; // more than one line selected, no need to adapt region
			} else {
				if (this.scannerEndPosition - 1 == lineEnd) { // EOF reached?
					return false;
				}
				return true; // a single line was selected
			}
		}
		return false;
	}
	
	private boolean isOnFirstColumn(int start) {
		if (this.lineEnds == null) return start == 0;
		int index = Arrays.binarySearch(this.lineEnds, start);
		// we want the line end of the previous line
		int previousLineEnd = this.getLineEnd(-index - 1);
		return previousLineEnd != -1 && previousLineEnd == start - 1;
	}

	private boolean isValidEdit(OptimizedReplaceEdit edit) {
		final int editLength= edit.length;
		final int editReplacementLength= edit.replacement.length();
		final int editOffset= edit.offset;
		if (editLength != 0) {
			
			IRegion covering = getCoveringRegion(editOffset, (editOffset + editLength - 1));
			if (covering != null) {
				if (editReplacementLength != 0 && editLength == editReplacementLength) {
					for (int i = editOffset, max = editOffset + editLength; i < max; i++) {
						if (scanner.source[i] != edit.replacement.charAt(i - editOffset)) {
							return true;
						}
					}
					return false;
				}
				return true;
			}

			IRegion starting = getRegionAt(editOffset + editLength);
			if (starting != null) {
				int i = editOffset;
				for (int max = editOffset + editLength; i < max; i++) {
					int replacementStringIndex = i - editOffset;
					if (replacementStringIndex >= editReplacementLength || scanner.source[i] != edit.replacement.charAt(replacementStringIndex)) {
						break;
					}
				}
				if (i - editOffset != editReplacementLength && i != editOffset + editLength - 1) {
					edit.offset = starting.getOffset();
					edit.length = 0;
					edit.replacement = edit.replacement.substring(i - editOffset);
					return true;
				}
			}
			
			return false;
		}
		
		IRegion covering = getCoveringRegion(editOffset, editOffset);
		if (covering != null) {
			return true;
		}

		if (editOffset == this.scannerEndPosition) {
			int index = Arrays.binarySearch(
				this.adaptedRegions,
				new Region(editOffset, 0),
				new Comparator() {
					public int compare(Object o1, Object o2) {
						IRegion r1 = (IRegion)o1;
						IRegion r2 = (IRegion)o2;
						
						int r1End = r1.getOffset() + r1.getLength();
						int r2End = r2.getOffset() + r2.getLength();
						
						return r1End - r2End;
					}
				});
			if (index < 0) {
				return false;
			}
			return true;
		}
		return false;
	}

	private void preserveEmptyLines(int count, int insertPosition) {
		if (count > 0) {
			if (this.formatter.preferences.number_of_empty_lines_to_preserve != 0) {
				int linesToPreserve = Math.min(count, this.formatter.preferences.number_of_empty_lines_to_preserve);
				this.printEmptyLines(linesToPreserve, insertPosition);
			} else {
				printNewLine(insertPosition);
			}
		}
	}

	private void print(int length, boolean considerSpaceIfAny) {
		if (checkLineWrapping && length + column > this.pageWidth) {
			handleLineTooLong();
		}
		this.lastNumberOfNewLines = 0;
		if (this.indentationLevel != 0) {
			printIndentationIfNecessary();
		}
		if (considerSpaceIfAny) {
			this.space();
		}
		if (this.pendingSpace) {
			this.addInsertEdit(this.scanner.getCurrentTokenStartPosition(), " "); //$NON-NLS-1$
		}
		this.pendingSpace = false;	
		this.needSpace = false;		
		column += length;
		needSpace = true;
	}

	private void printBlockComment(boolean isJavadoc) {
		int currentTokenStartPosition = this.scanner.getCurrentTokenStartPosition();
		int currentTokenEndPosition = this.scanner.getCurrentTokenEndPosition() + 1;
		
		this.scanner.resetTo(currentTokenStartPosition, currentTokenEndPosition - 1);
		int currentCharacter;
		boolean isNewLine = false;
		int start = currentTokenStartPosition;
		int nextCharacterStart = currentTokenStartPosition;
		int previousStart = currentTokenStartPosition;
		boolean onFirstColumn = isOnFirstColumn(start);

		boolean indentComment = false;
		if (this.indentationLevel != 0) {
			if (isJavadoc
					|| !this.formatter.preferences.never_indent_block_comments_on_first_column
					|| !onFirstColumn) {
				indentComment = true;
				printIndentationIfNecessary();
			}
		}
		if (this.pendingSpace) {
			this.addInsertEdit(currentTokenStartPosition, " "); //$NON-NLS-1$
		}
		this.needSpace = false;
		this.pendingSpace = false;
		
		if (includesBlockComments()) {
			printBlockComment(currentTokenStartPosition, currentTokenEndPosition);
			return;
		}

		int currentCommentOffset = onFirstColumn ? 0 : getCurrentCommentOffset(start);
		boolean formatComment = (isJavadoc && (this.formatComments & CodeFormatter.K_JAVA_DOC) != 0) || (!isJavadoc && (this.formatComments & CodeFormatter.K_MULTI_LINE_COMMENT) != 0);

		while (nextCharacterStart <= currentTokenEndPosition && (currentCharacter = this.scanner.getNextChar()) != -1) {
			nextCharacterStart = this.scanner.currentPosition;

			switch(currentCharacter) {
				case '\r' :
					start = previousStart;
					isNewLine = true;
					if (this.scanner.getNextChar('\n')) {
						currentCharacter = '\n';
						nextCharacterStart = this.scanner.currentPosition;
					}
					break;
				case '\n' :
					start = previousStart;
					isNewLine = true;
					nextCharacterStart = this.scanner.currentPosition;
					break;
				default:
					if (isNewLine) {
						this.column = 1;
						this.line++;
						isNewLine = false;
						
						StringBuffer buffer = new StringBuffer();
						if (onFirstColumn) {
							// simply insert indentation if necessary
							buffer.append(this.lineSeparator);
							if (indentComment) {
								printIndentationIfNecessary(buffer);
							}
							if (formatComment) {
								if (ScannerHelper.isWhitespace((char) currentCharacter)) {
									int previousStartPosition = this.scanner.currentPosition;
									while(currentCharacter != -1 && currentCharacter != '\r' && currentCharacter != '\n' && ScannerHelper.isWhitespace((char) currentCharacter)) {
										previousStart = nextCharacterStart;
										previousStartPosition = this.scanner.currentPosition;
										currentCharacter = this.scanner.getNextChar();
										nextCharacterStart = this.scanner.currentPosition;
									}
									if (currentCharacter == '\r' || currentCharacter == '\n') {
										nextCharacterStart = previousStartPosition;
									}
								}
								if (currentCharacter != '\r' && currentCharacter != '\n') {
									buffer.append(' ');
								}
							}
						} else {
							if (ScannerHelper.isWhitespace((char) currentCharacter)) {
								int previousStartPosition = this.scanner.currentPosition;
								int count = 0;
								loop: while(currentCharacter != -1 && currentCharacter != '\r' && currentCharacter != '\n' && ScannerHelper.isWhitespace((char) currentCharacter)) {
									if (count >= currentCommentOffset) {
										break loop;
									}
									previousStart = nextCharacterStart;
									previousStartPosition = this.scanner.currentPosition;
									switch(currentCharacter) {
										case '\t' :
											count += this.tabLength;
											break;
										default :
											count ++;
									}
									currentCharacter = this.scanner.getNextChar();
									nextCharacterStart = this.scanner.currentPosition;
								}
								if (currentCharacter == '\r' || currentCharacter == '\n') {
									nextCharacterStart = previousStartPosition;
								}
							}
							buffer.append(this.lineSeparator);
							if (indentComment) {
								printIndentationIfNecessary(buffer);
							}
							if (formatComment) {
								int previousStartTemp = previousStart;
								int nextCharacterStartTemp = nextCharacterStart;
								while(currentCharacter != -1 && currentCharacter != '\r' && currentCharacter != '\n' && ScannerHelper.isWhitespace((char) currentCharacter)) {
									previousStart = nextCharacterStart;
									currentCharacter = this.scanner.getNextChar();
									nextCharacterStart = this.scanner.currentPosition;
								}
								if (currentCharacter == '*') {
									buffer.append(' ');
								} else {
									previousStart = previousStartTemp;
									nextCharacterStart = nextCharacterStartTemp;
								}
								this.scanner.currentPosition = nextCharacterStart;
							}
						}
						addReplaceEdit(start, previousStart - 1, String.valueOf(buffer));
					} else {
						this.column += (nextCharacterStart - previousStart);
					}
			}
			previousStart = nextCharacterStart;
			this.scanner.currentPosition = nextCharacterStart;
		}
		this.lastNumberOfNewLines = 0;
		needSpace = false;
		this.scanner.resetTo(currentTokenEndPosition, this.scannerEndPosition - 1);
	}

	private void printBlockComment(int currentTokenStartPosition, int currentTokenEndPosition) {


		// Compute indentation
		int maxColumn = this.formatter.preferences.comment_line_length + 1;
		int indentLevel = this.indentationLevel;
		int indentations = this.numberOfIndentations;
		this.indentationLevel = (this.column / this.tabLength) * this.tabLength;
		this.column = this.indentationLevel + 1;
		this.numberOfIndentations = this.indentationLevel / this.indentationSize;

		// Consume the comment prefix
		StringBuffer buffer = new StringBuffer();
		this.scanner.getNextChar();
		this.scanner.getNextChar();
		this.column += 2;
		this.scanner.skipComments = true;
		StringBuffer tokensBuffer = new StringBuffer();

		// Consume text token per token
		int previousToken = -1;
		boolean newLine = false;
		boolean multiLines = false;
		boolean hasMultiLines = false;
		boolean hasTokens = false;
		boolean bufferHasTokens = false;
		int hasTextOnFirstLine = 0;
		boolean firstWord = true;
		boolean clearBlankLines = this.formatter.preferences.comment_clear_blank_lines_in_block_comment;
		int scannerLine = Util.getLineNumber(this.scanner.currentPosition, this.lineEnds, 0, this.maxLines);
		int firstLine = scannerLine;
		int lineNumber = scannerLine;
		int lastTextLine = -1;
		boolean openedString = false;
		while (!this.scanner.atEnd()) {
			
			// Consume token
			int token;
			try {
				token = this.scanner.getNextToken();
			} catch (InvalidInputException iie) {
				String msg = iie.getMessage();
				boolean insertSpace = (previousToken == TerminalTokens.TokenNameWHITESPACE || newLine) && !firstWord;
				if (msg == Scanner.INVALID_CHARACTER_CONSTANT) {
					if (insertSpace) {
						tokensBuffer.append(' ');
					}
					tokensBuffer.append('\'');
				} else if (msg == Scanner.INVALID_CHAR_IN_STRING) {
					if (openedString) {
						openedString = false;
					} else {
						if (insertSpace) {
							tokensBuffer.append(' ');
						}
						openedString = true;
					}
					tokensBuffer.append('"');
				} else {
					// skip failure
				}
				// Need to retrieve correct position
				this.scanner.resetTo(this.scanner.startPosition, currentTokenEndPosition-1);
				this.scanner.getNextChar();
				previousToken = 2000;
				newLine = false;
				continue;
			}
			
			// Look at specific tokens
    		boolean insertSpace = (previousToken == TerminalTokens.TokenNameWHITESPACE) && (!firstWord || !hasTokens);
			switch (token) {
				case TerminalTokens.TokenNameWHITESPACE:
					if (tokensBuffer.length() > 0) {
						if (hasTextOnFirstLine == 1 && multiLines) {
							printBlockCommentHeaderLine(buffer);
							hasTextOnFirstLine = -1;
						}
						buffer.append(tokensBuffer);
						this.column += tokensBuffer.length();
						tokensBuffer.setLength(0);
						bufferHasTokens = true;
					}
					if (previousToken == -1) {
						// do not remember the first whitespace
						previousToken = -2;
					} else {
						previousToken = token;
					}
					lineNumber = Util.getLineNumber(this.scanner.getCurrentTokenEndPosition(), this.lineEnds, scannerLine>1 ? scannerLine-2 : 0, this.maxLines);
					if (lineNumber > scannerLine) {
						hasMultiLines = true;
						newLine = true;
					}
					scannerLine = lineNumber;
					continue;
				case TerminalTokens.TokenNameMULTIPLY:
					previousToken = token;
					lineNumber = Util.getLineNumber(this.scanner.getCurrentTokenEndPosition(), this.lineEnds, scannerLine>1 ? scannerLine-2 : 0, this.maxLines);
					if (this.scanner.currentCharacter == '/') {
						// Add remaining buffered tokens
						if (tokensBuffer.length() > 0) {
							buffer.append(tokensBuffer);
							this.column += tokensBuffer.length();
						}
						// end of comment
						if (multiLines || hasMultiLines) {
					    	buffer.append(this.lineSeparator);
					    	this.column = 1;
					    	printIndentationIfNecessary(buffer);
						}
						buffer.append(' ');
			    		buffer.append(BLOCK_FOOTER);
				    	this.column += BLOCK_FOOTER_LENGTH + 1;
				    	this.scanner.getNextChar(); // reach the end of scanner
				    	continue;
					}
					scannerLine = lineNumber;
					continue;
				case TerminalTokens.TokenNameMULTIPLY_EQUAL:
					if (newLine) {
						this.scanner.resetTo(this.scanner.startPosition, currentTokenEndPosition-1);
						this.scanner.getNextChar(); // consume the multiply
						previousToken = TerminalTokens.TokenNameMULTIPLY;
						scannerLine = Util.getLineNumber(this.scanner.getCurrentTokenEndPosition(), this.lineEnds, scannerLine>1 ? scannerLine-2 : 0, this.maxLines);
						continue;
					}
			}

			// Look at gap and insert corresponding lines if necessary
			int linesGap;
			int max;
			lineNumber = Util.getLineNumber(this.scanner.getCurrentTokenEndPosition(), this.lineEnds, scannerLine>1 ? scannerLine-2 : 0, this.maxLines);
			if (lastTextLine == -1) {
				linesGap = lineNumber - firstLine;
				max = 0;
			} else {
				linesGap = lineNumber - lastTextLine;
				if (token == TerminalTokens.TokenNameAT && linesGap ==1) {
					// insert one blank line before root tags
					linesGap = 2;
				}
				max = 1;
			}
			if (linesGap > max) {
				if (clearBlankLines) {
					// TODO (frederic) see if there's a bug for the unremoved blank line for root tags
					 if (token == TerminalTokens.TokenNameAT) {
						 linesGap = 1;
					 } else {
						linesGap = max==0 ? 1 : 0;
					 }
				}
				for (int i=0; i<linesGap; i++) {
					// Add remaining buffered tokens
					if (tokensBuffer.length() > 0) {
						if (hasTextOnFirstLine == 1) {
							printBlockCommentHeaderLine(buffer);
							hasTextOnFirstLine = -1;
						}
						buffer.append(tokensBuffer);
						tokensBuffer.setLength(0);
						bufferHasTokens = true;
					}
			    	buffer.append(this.lineSeparator);
			    	this.column = 1;
			    	printIndentationIfNecessary(buffer);
		    		buffer.append(BLOCK_LINE_PREFIX);
		    		this.column += BLOCK_LINE_PREFIX_LENGTH;
		    		firstWord = true;
					multiLines = true;
				}
				insertSpace = insertSpace && linesGap == 0;
			}

			// Increment column
			int tokenStart = this.scanner.getCurrentTokenStartPosition();
    		int tokenLength = (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - tokenStart;
    		hasTokens = true;
    		if (hasTextOnFirstLine == 0) {
    			if (firstLine == lineNumber) {
	    			hasTextOnFirstLine = 1;
	    			this.column++; // include first space
	    		} else {
	    			hasTextOnFirstLine = -1;
	    		}
    		}
    		int lastColumn = this.column + tokensBuffer.length() + tokenLength;
    		if (insertSpace) lastColumn++;

    		// Append next token inserting a new line if max line is reached
			if (!firstWord && lastColumn > maxColumn) {
		    	String tokensString = tokensBuffer.toString().trim();
				// not enough space on the line
				if (hasTextOnFirstLine == 1) {
					printBlockCommentHeaderLine(buffer);
				}
				if ((this.indentationLevel+tokensString.length()+tokenLength) > maxColumn) {
					// there won't be enough room even if we break the line before the buffered tokens
					// So add the buffered tokens now
					buffer.append(tokensString);
					this.column += tokensString.length();
					tokensBuffer.setLength(0);
				}
				if (bufferHasTokens) {
			    	buffer.append(this.lineSeparator);
			    	this.column = 1;
			    	printIndentationIfNecessary(buffer);
		    		buffer.append(BLOCK_LINE_PREFIX);
			    	this.column += BLOCK_LINE_PREFIX_LENGTH;
				}
		    	if (tokensBuffer.length() > 0) {
					buffer.append(tokensString);
					this.column += tokensString.length();
					tokensBuffer.setLength(0);
		    	}
				buffer.append(this.scanner.source, tokenStart, tokenLength);
				bufferHasTokens = true;
				this.column += tokenLength;
				multiLines = true;
				hasTextOnFirstLine = -1;
			} else {
				// append token to the line
				if (insertSpace)  {
					tokensBuffer.append(' ');
				}
				tokensBuffer.append(this.scanner.source, tokenStart, tokenLength);
			}
			previousToken = token;
			newLine = false;
    		firstWord = false;
			scannerLine = lineNumber;
			lastTextLine = lineNumber;
		}

		// Replace block comment text
		if (hasTokens || multiLines) {
			StringBuffer replacement = new StringBuffer(BLOCK_HEADER);
			if (hasTextOnFirstLine == 1) {
				if ((hasMultiLines || multiLines)) {
					int col = this.column;
					replacement.append(this.lineSeparator);
					this.column = 1;
					printIndentationIfNecessary(replacement);
					replacement.append(BLOCK_LINE_PREFIX);
			    	this.column = col;
				} else {
					replacement.append(' ');
				}
			}
			replacement.append(buffer);
			addReplaceEdit(currentTokenStartPosition, currentTokenEndPosition-1, replacement.toString());
		}

		// Reset
		this.indentationLevel = indentLevel;
		this.numberOfIndentations = indentations;
		this.lastNumberOfNewLines = 0;
		needSpace = false;
		this.scanner.resetTo(currentTokenEndPosition, this.scannerEndPosition - 1);
		this.scanner.skipComments = false;
	}

	private void printBlockCommentHeaderLine(StringBuffer buffer) {
	    if (buffer.length() == 0) {
	    	buffer.append(this.lineSeparator);
	    	this.column = 1;
	    	printIndentationIfNecessary(buffer);
	    	buffer.append(BLOCK_LINE_PREFIX);
	    	this.column += BLOCK_LINE_PREFIX_LENGTH;
	    } else {
	    	StringBuffer insert = new StringBuffer();
	    	insert.append(this.lineSeparator);
	    	this.column = 1;
	    	printIndentationIfNecessary(insert);
	    	insert.append(BLOCK_LINE_PREFIX);
	    	this.column += BLOCK_LINE_PREFIX_LENGTH;
	    	buffer.insert(0, insert.toString());
	    }
    }

	public void printEndOfCompilationUnit() {
		try {
			// if we have a space between two tokens we ensure it will be dumped in the formatted string
			int currentTokenStartPosition = this.scanner.currentPosition;
			boolean hasComment = false;
			boolean hasLineComment = false;
			boolean hasWhitespace = false;
			int count = 0;
			while (true) {
				this.currentToken = this.scanner.getNextToken();
				switch(this.currentToken) {
					case TerminalTokens.TokenNameWHITESPACE :
						char[] whiteSpaces = this.scanner.getCurrentTokenSource();
						count = 0;
						for (int i = 0, max = whiteSpaces.length; i < max; i++) {
							switch(whiteSpaces[i]) {
								case '\r' :
									if ((i + 1) < max) {
										if (whiteSpaces[i + 1] == '\n') {
											i++;
										}
									}
									count++;
									break;
								case '\n' :
									count++;
							}
						}
						if (count == 0) {
							hasWhitespace = true;
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						} else if (hasComment) {
							if (count == 1) {
								this.printNewLine(this.scanner.getCurrentTokenStartPosition());
							} else {
								preserveEmptyLines(count - 1, this.scanner.getCurrentTokenStartPosition());
							}
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						} else if (hasLineComment) {
							this.preserveEmptyLines(count, this.scanner.getCurrentTokenStartPosition());
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						} else {
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						}
						currentTokenStartPosition = this.scanner.currentPosition;						
						break;
					case TerminalTokens.TokenNameCOMMENT_LINE :
						if (count >= 1) {
							if (count > 1) {
								preserveEmptyLines(count - 1, this.scanner.getCurrentTokenStartPosition());
							} else if (count == 1) {
								printNewLine(this.scanner.getCurrentTokenStartPosition());
							}
						} else if (hasWhitespace) {
							space();
						} 
						hasWhitespace = false;
						printLineComment();
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = true;		
						count = 0;
						break;
					case TerminalTokens.TokenNameCOMMENT_BLOCK :
						if (count >= 1) {
							if (count > 1) {
								preserveEmptyLines(count - 1, this.scanner.getCurrentTokenStartPosition());
							} else if (count == 1) {
								printNewLine(this.scanner.getCurrentTokenStartPosition());
							}
						} else if (hasWhitespace) {
							space();
						} 
						hasWhitespace = false;
						printBlockComment(false);
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = false;
						hasComment = true;
						count = 0;
						break;
					case TerminalTokens.TokenNameCOMMENT_JAVADOC :
						if (count >= 1) {
							if (count > 1) {
								preserveEmptyLines(count - 1, this.scanner.startPosition);
							} else if (count == 1) {
								printNewLine(this.scanner.startPosition);
							}
						} else if (hasWhitespace) {
							space();
						} 
						hasWhitespace = false;
						if (includesJavadocComments()) {
							printJavadocComment(this.scanner.startPosition, this.scanner.currentPosition);
						} else {
							printBlockComment(true);
						}
						printNewLine();
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = false;
						hasComment = true;
						count = 0;
						break;
					case TerminalTokens.TokenNameSEMICOLON :
						print(this.scanner.currentPosition - this.scanner.startPosition, this.formatter.preferences.insert_space_before_semicolon);
						break;
					case TerminalTokens.TokenNameEOF :
						if (count >= 1 || this.formatter.preferences.insert_new_line_at_end_of_file_if_missing) {
							this.printNewLine(this.scannerEndPosition);
						}
						return;
					default :
						// step back one token
						this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
						return;
				}
			}
		} catch (InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}
	/*
	 * prints a code snippet
	 */
	private void printCodeSnippet(int startPosition, int endPosition) {
		String snippet = new String(this.scanner.source, startPosition, endPosition - startPosition + 1);
		
		// 1 - strip content prefix (@see JavaDocRegion#preprocessCodeSnippet)
		ILineTracker tracker= new DefaultLineTracker();
		String contentPrefix= IJavaDocTagConstants.JAVADOC_STAR;

		StringBuffer inputBuffer= new StringBuffer();
		inputBuffer.setLength(0);
		inputBuffer.append(snippet);
		tracker.set(snippet);
		for (int lines= tracker.getNumberOfLines() - 1; lines > 0; lines--) {
			int lineOffset;
			try {
				lineOffset= tracker.getLineOffset(lines);
			} catch (BadLocationException e) {
				// should not happen
				CommentFormatterUtil.log(e);
				return;
			}
			int prefixOffset= inputBuffer.indexOf(contentPrefix, lineOffset);
			if (prefixOffset >= 0 && inputBuffer.substring(lineOffset, prefixOffset).trim().length() == 0)
				inputBuffer.delete(lineOffset, prefixOffset + 1 + 1);
		}
		
		// 2 - convert HTML to Java (@see JavaDocRegion#convertHtml2Java)
		HTMLEntity2JavaReader reader= new HTMLEntity2JavaReader(new StringReader(inputBuffer.toString()));
		char[] buf= new char[snippet.length()]; // html2text never gets longer, only shorter!
		String convertedSnippet;
		try {
			int read= reader.read(buf);
			convertedSnippet = new String(buf, 0, read);
		} catch (IOException e) {
			// should not happen
			CommentFormatterUtil.log(e);
			return;
		}
		
		// 3 - process snippet (@see JavaDocRegion#formatCodeSnippet)
		String formattedSnippet = convertedSnippet;
		TextEdit edit= CommentFormatterUtil.format2(CodeFormatter.K_UNKNOWN, convertedSnippet, 0, this.lineSeparator, this.formatter.preferences.getMap());
		if (edit != null) {
			formattedSnippet= CommentFormatterUtil.evaluateFormatterEdit(convertedSnippet, edit, null);
		}

		// remove trailing spaces
		formattedSnippet= formattedSnippet.trim();

		// 4 - add the content prefix (@see JavaDocRegion#postprocessCodeSnippet)
		StringBuffer outputBuffer= new StringBuffer();
		tracker= new DefaultLineTracker();
		this.column = 1;
		printIndentationIfNecessary(outputBuffer); // append indentation
		outputBuffer.append(BLOCK_LINE_PREFIX);
		String linePrefix = outputBuffer.toString();		
		outputBuffer.setLength(0);
		
		// 5 - convert back to HTML (@see JavaDocRegion#convertJava2Html)
		Java2HTMLEntityReader javaReader= new Java2HTMLEntityReader(new StringReader(formattedSnippet));
		buf= new char[256];
		StringBuffer conversionBuffer= new StringBuffer();
		int l;
		try {
			do {
				l= javaReader.read(buf);
				if (l != -1)
					conversionBuffer.append(buf, 0, l);
			} while (l > 0);
			formattedSnippet = conversionBuffer.toString();
		} catch (IOException e) {
			// should not happen
			CommentFormatterUtil.log(e);
			return;
		}
		
		outputBuffer.append(formattedSnippet);
		
		tracker.set(outputBuffer.toString());
		
		for (int lines= tracker.getNumberOfLines() - 1; lines > 0; lines--) {
			try {
				outputBuffer.insert(tracker.getLineOffset(lines), linePrefix);
			} catch (BadLocationException e) {
				// should not happen
				CommentFormatterUtil.log(e);
				return;
			}
		}
		// replace old text with the formatted snippet
		addReplaceEdit(startPosition, endPosition, outputBuffer.toString());		
	}

	void printComment() {
		printComment(CodeFormatter.K_UNKNOWN);
	}
	
	/*
	 * Main method to print and format comments (javadoc, block and single line comments)
	 */
	void printComment(int kind) {
		final boolean rejectLineComment = kind  == CodeFormatter.K_MULTI_LINE_COMMENT || kind == CodeFormatter.K_JAVA_DOC;
		final boolean rejectBlockComment = kind  == CodeFormatter.K_SINGLE_LINE_COMMENT || kind  == CodeFormatter.K_JAVA_DOC;
		final boolean rejectJavadocComment = kind  == CodeFormatter.K_SINGLE_LINE_COMMENT || kind  == CodeFormatter.K_MULTI_LINE_COMMENT;
		try {
			// if we have a space between two tokens we ensure it will be dumped in the formatted string
			int currentTokenStartPosition = this.scanner.currentPosition;
			boolean hasComment = false;
			boolean hasLineComment = false;
			boolean hasWhitespace = false;
			int count = 0;
			while ((this.currentToken = this.scanner.getNextToken()) != TerminalTokens.TokenNameEOF) {
				switch(this.currentToken) {
					case TerminalTokens.TokenNameWHITESPACE :
						char[] whiteSpaces = this.scanner.getCurrentTokenSource();
						count = 0;
						for (int i = 0, max = whiteSpaces.length; i < max; i++) {
							switch(whiteSpaces[i]) {
								case '\r' :
									if ((i + 1) < max) {
										if (whiteSpaces[i + 1] == '\n') {
											i++;
										}
									}
									count++;
									break;
								case '\n' :
									count++;
							}
						}
						if (count == 0) {
							hasWhitespace = true;
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						} else if (hasComment) {
							if (count == 1) {
								this.printNewLine(this.scanner.getCurrentTokenStartPosition());
							} else {
								preserveEmptyLines(count - 1, this.scanner.getCurrentTokenStartPosition());
							}
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						} else if (hasLineComment) {
							this.preserveEmptyLines(count, this.scanner.getCurrentTokenStartPosition());
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						} else if (count != 0 && this.formatter.preferences.number_of_empty_lines_to_preserve != 0) {
							addReplaceEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition(), this.getPreserveEmptyLines(count - 1));
						} else {
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						}
						currentTokenStartPosition = this.scanner.currentPosition;						
						break;
					case TerminalTokens.TokenNameCOMMENT_LINE :
						if (rejectLineComment) break;
						if (count >= 1) {
							if (count > 1) {
								preserveEmptyLines(count - 1, this.scanner.getCurrentTokenStartPosition());
							} else if (count == 1) {
								printNewLine(this.scanner.getCurrentTokenStartPosition());
							}
						} else if (hasWhitespace) {
							space();
						} 
						hasWhitespace = false;
						printLineComment();
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = true;		
						count = 0;
						break;
					case TerminalTokens.TokenNameCOMMENT_BLOCK :
						if (rejectBlockComment) break;
						if (count >= 1) {
							if (count > 1) {
								preserveEmptyLines(count - 1, this.scanner.getCurrentTokenStartPosition());
							} else if (count == 1) {
								printNewLine(this.scanner.getCurrentTokenStartPosition());
							}
						} else if (hasWhitespace) {
							space();
						} 
						hasWhitespace = false;
						printBlockComment(false);
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = false;
						hasComment = true;
						count = 0;
						break;
					case TerminalTokens.TokenNameCOMMENT_JAVADOC :
						if (rejectJavadocComment) break;
						if (count >= 1) {
							if (count > 1) {
								preserveEmptyLines(count - 1, this.scanner.getCurrentTokenStartPosition());
							} else if (count == 1) {
								printNewLine(this.scanner.getCurrentTokenStartPosition());
							}
						} else if (hasWhitespace) {
							space();
						} 
						hasWhitespace = false;
						if (includesJavadocComments()) {
							printJavadocComment(this.scanner.startPosition, this.scanner.currentPosition);
						} else {
							printBlockComment(true);
						}
						printNewLine();
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = false;
						hasComment = true;
						count = 0;
						break;
					default :
						// step back one token
						this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
						return;
				}
			}
		} catch (InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}

	void printComment(int kind, String source, int start, int end, int level) {

		// Set scanner
		initializeScanner(source.toCharArray());
		this.scanner.resetTo(start, end);

		// Set indentation level
	    this.numberOfIndentations = level;
	    this.indentationLevel = level * this.indentationSize;
	    this.column = this.indentationLevel + 1;

	    // Print corresponding comment
	    switch (kind) {
	    	case CodeFormatter.K_SINGLE_LINE_COMMENT:
			    printComment(kind);
	    		break;
	    	case CodeFormatter.K_MULTI_LINE_COMMENT:
			    printComment(kind);
	    		break;
	    	case CodeFormatter.K_JAVA_DOC:
	    		printJavadocComment(start, end);
	    		break;
	    }
    }
	
	private void printLineComment() {
    	int currentTokenStartPosition = this.scanner.getCurrentTokenStartPosition();
    	int currentTokenEndPosition = this.scanner.getCurrentTokenEndPosition() + 1;
    	boolean isNlsTag = false;
    	if (CharOperation.indexOf(Scanner.TAG_PREFIX, this.scanner.source, true, currentTokenStartPosition, currentTokenEndPosition) != -1) {
    		this.nlsTagCounter = 0;
    		isNlsTag = true;
    	}
    	this.scanner.resetTo(currentTokenStartPosition, currentTokenEndPosition - 1);
    	int currentCharacter;
    	int start = currentTokenStartPosition;
    	int nextCharacterStart = currentTokenStartPosition;
    	
    	if (this.indentationLevel != 0) {
    		if (!this.formatter.preferences.never_indent_line_comments_on_first_column
    				|| !isOnFirstColumn(start)) {
    			printIndentationIfNecessary();
    		}
    	}
    	if (this.pendingSpace) {
    		this.addInsertEdit(currentTokenStartPosition, " "); //$NON-NLS-1$
    	}
    	this.needSpace = false;
    	this.pendingSpace = false;
    	int previousStart = currentTokenStartPosition;

		if (!isNlsTag && includesLineComments()) {
			printLineComment(currentTokenStartPosition, currentTokenEndPosition-1);
		} else {
			// do nothing!?
	    	loop: while (nextCharacterStart <= currentTokenEndPosition && (currentCharacter = this.scanner.getNextChar()) != -1) {
	    		nextCharacterStart = this.scanner.currentPosition;
	    
	    		switch(currentCharacter) {
	    			case '\r' :
	    				start = previousStart;
	    				break loop;
	    			case '\n' :
	    				start = previousStart;
	    				break loop;
	    		}
	    		previousStart = nextCharacterStart;
	    	}
	    	if (start != currentTokenStartPosition) {
	    		// this means that the line comment doesn't end the file
	    		addReplaceEdit(start, currentTokenEndPosition - 1, lineSeparator);
	    		this.line++; 
	    		this.column = 1;
	    		this.lastNumberOfNewLines = 1;
	    	}
		}
    	this.needSpace = false;
    	this.pendingSpace = false;
    	// realign to the proper value
    	if (this.currentAlignment != null) {
    		if (this.memberAlignment != null) {
    			// select the last alignment
    			if (this.currentAlignment.location.inputOffset > this.memberAlignment.location.inputOffset) {
    				if (this.currentAlignment.couldBreak() && this.currentAlignment.wasSplit) {
    					this.currentAlignment.performFragmentEffect();
    				}
    			} else {
    				this.indentationLevel = Math.max(this.indentationLevel, this.memberAlignment.breakIndentationLevel);
    			}
    		} else if (this.currentAlignment.couldBreak() && this.currentAlignment.wasSplit) {
    			this.currentAlignment.performFragmentEffect();
    		}
    	}
    	this.scanner.resetTo(currentTokenEndPosition, this.scannerEndPosition - 1);
    }

	private void printLineComment(int commentStart, int commentEnd) {

		// Compute indentation
		int firstColumn = this.column;
		int indentLevel = this.indentationLevel;
		int indentations = this.numberOfIndentations;
		this.indentationLevel = getNextIndentationLevel(firstColumn);
		this.numberOfIndentations = this.indentationLevel / this.indentationSize;

		// Consume the comment prefix
		this.scanner.resetTo(commentStart, commentEnd);
		StringBuffer buffer = new StringBuffer();
		this.scanner.getNextChar();
		buffer.append(this.scanner.currentCharacter);
		this.scanner.getNextChar();
		buffer.append(this.scanner.currentCharacter);
		this.column += 2;

		// Scan the text token per token to compact it and size it the max line length
		int maxColumn = this.formatter.preferences.comment_line_length + 1;
		int previousToken = -1;
		int previousPosition = commentStart;
		char previousChar = 0;
		boolean firstWord = true;
		this.scanner.skipComments = true;

		// Consume text token per token
		while (!this.scanner.atEnd()) {
			int token;
			try {
				token = this.scanner.getNextToken();
			} catch (InvalidInputException iie) {
				String msg = iie.getMessage();
				if (msg == Scanner.INVALID_CHARACTER_CONSTANT) {
					buffer.append('\'');
				} else if (msg == Scanner.INVALID_CHAR_IN_STRING) {
					buffer.append('"');
				} else {
					// skip failure
				}
				// Need to retrieve correct position
				this.scanner.resetTo(this.scanner.startPosition, commentEnd);
				this.scanner.getNextChar();
				this.column++;
				continue;
			}
			switch (token) {
				case TerminalTokens.TokenNameWHITESPACE:
					previousToken = token;
					previousPosition = this.scanner.currentPosition;
					previousChar = this.scanner.currentCharacter;
					continue;
				case TerminalTokens.TokenNameEOF:
					continue;
			}
			int tokenStart = this.scanner.getCurrentTokenStartPosition();
    		int tokenLength = (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - tokenStart;
    		boolean insertSpace = previousToken == TerminalTokens.TokenNameWHITESPACE;
			this.column += tokenLength;
    		if (previousToken == -1 || insertSpace) this.column++;
    		int col = column; 
			if (!firstWord && col > maxColumn) {
				// not enough space on the line
				this.line++;
		    	buffer.append(this.lineSeparator);
		    	this.column = 1;
		    	printIndentationIfNecessary(buffer);
	    		buffer.append(LINE_COMMENT_PREFIX);
		    	this.column = firstColumn + LINE_COMMENT_PREFIX_LENGTH;
				buffer.append(this.scanner.source, tokenStart, tokenLength);
				this.column += tokenLength;
			} else {
				// append token to the line
				if (previousToken == -1 || insertSpace)  {
	    			buffer.append(' ');
				}
				buffer.append(this.scanner.source, tokenStart, tokenLength);
			}
			previousToken = token;
			previousPosition = this.scanner.currentPosition;
			previousChar = this.scanner.currentCharacter;
			firstWord = false;
		}

		// Append separator if the comment is not at the end of file
		this.indentationLevel = indentLevel;
		this.numberOfIndentations = indentations;
		this.lastNumberOfNewLines = 0;
		if (previousChar == '\n' || previousChar == '\r') {
			// line comment is normally ended with new line
			this.column = 1;
			buffer.append(this.lineSeparator);
			this.line++;
			this.lastNumberOfNewLines++;
		} else {
			this.scanner.resetTo(previousPosition, commentEnd);
			while (!this.scanner.atEnd()) {
				this.scanner.getNextChar();
				if (this.scanner.currentCharacter == '\n' || this.scanner.currentCharacter == '\r') {
					// line comment is normally ended with new line
					buffer.append(this.lineSeparator);
					this.column = 1;
					this.line++;
					this.lastNumberOfNewLines++;
					break;
				}
			}
		}

		// Replace the existing comment with new one
		addReplaceEdit(commentStart, commentEnd, buffer.toString());
		this.scanner.skipComments = false;
	}

	public void printEmptyLines(int linesNumber) {
		this.printEmptyLines(linesNumber, this.scanner.getCurrentTokenEndPosition() + 1);
	}

	private void printEmptyLines(int linesNumber, int insertPosition) {
		final String buffer = getEmptyLines(linesNumber);
		if (Util.EMPTY_STRING == buffer) return;
		addInsertEdit(insertPosition, buffer);
	}

	void printIndentationIfNecessary() {
		StringBuffer buffer = new StringBuffer();
		printIndentationIfNecessary(buffer);
		if (buffer.length() > 0) {
			addInsertEdit(this.scanner.getCurrentTokenStartPosition(), buffer.toString());
			this.pendingSpace = false;
		}
	}

	private void printIndentationIfNecessary(StringBuffer buffer) {
		switch(this.tabChar) {
			case DefaultCodeFormatterOptions.TAB :
				boolean useTabsForLeadingIndents = this.useTabsOnlyForLeadingIndents;
				int numberOfLeadingIndents = this.numberOfIndentations;
				int indentationsAsTab = 0;
				if (useTabsForLeadingIndents) {
					while (this.column <= this.indentationLevel) {
						if (indentationsAsTab < numberOfLeadingIndents) {
							if (buffer != null) buffer.append('\t');
							indentationsAsTab++;
							int complement = this.tabLength - ((this.column - 1) % this.tabLength); // amount of space
							this.column += complement;
							this.needSpace = false;
						} else {
							if (buffer != null) buffer.append(' ');
							this.column++;
							this.needSpace = false;
						}
					}
				} else {
					while (this.column <= this.indentationLevel) {
						if (buffer != null) buffer.append('\t');
						int complement = this.tabLength - ((this.column - 1) % this.tabLength); // amount of space
						this.column += complement;
						this.needSpace = false;
					}
				}
				break;
			case DefaultCodeFormatterOptions.SPACE :
				while (this.column <= this.indentationLevel) {
					if (buffer != null) buffer.append(' ');
					this.column++;
					this.needSpace = false;
				}
				break;
			case DefaultCodeFormatterOptions.MIXED :
				useTabsForLeadingIndents = this.useTabsOnlyForLeadingIndents;
				numberOfLeadingIndents = this.numberOfIndentations;
				indentationsAsTab = 0;
				if (useTabsForLeadingIndents) {
					final int columnForLeadingIndents = numberOfLeadingIndents * this.indentationSize;
					while (this.column <= this.indentationLevel) {
						if (this.column <= columnForLeadingIndents) {
							if ((this.column - 1 + this.tabLength) <= this.indentationLevel) {
								if (buffer != null) buffer.append('\t');
								this.column += this.tabLength;
							} else if ((this.column - 1 + this.indentationSize) <= this.indentationLevel) {
								// print one indentation
								for (int i = 0, max = this.indentationSize; i < max; i++) {
									if (buffer != null) buffer.append(' ');
									this.column++;
								}
							} else {
								if (buffer != null) buffer.append(' ');
								this.column++;
							}
						} else {
							for (int i = this.column, max = this.indentationLevel; i <= max; i++) {
								if (buffer != null) buffer.append(' ');
								this.column++;
							}
						}
						this.needSpace = false;
					}
				} else {
					while (this.column <= this.indentationLevel) {
						if ((this.column - 1 + this.tabLength) <= this.indentationLevel) {
							if (buffer != null) buffer.append('\t');
							this.column += this.tabLength;
						} else if ((this.column - 1 + this.indentationSize) <= this.indentationLevel) {
							// print one indentation
							for (int i = 0, max = this.indentationSize; i < max; i++) {
								if (buffer != null) buffer.append(' ');
								this.column++;
							}
						} else {
							if (buffer != null) buffer.append(' ');
							this.column++;
						}
						this.needSpace = false;
					}
				}
				break;
		}
	}

	private void printJavadocBlock(FormatJavadocBlock block) {
		if( block == null) return;

		// Init positions
		int previousEnd = block.tagEnd;
		int maxNodes = block.nodesPtr;

		// Compute indentation
		boolean headerLine = block.isHeaderLine() && this.lastNumberOfNewLines == 0;
		int firstColumn = 1 + this.indentationLevel + BLOCK_LINE_PREFIX_LENGTH;
		int maxColumn = this.formatter.preferences.comment_line_length + 1;
		if (headerLine) {
			firstColumn++;
			maxColumn++;
		}
		StringBuffer indentationBuffer = printJavadocIndentationBuffer(block, firstColumn);

		// format tag section if necessary
		boolean clearBlankLines = this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment;
		if (!block.isInlined()) {
			this.lastNumberOfNewLines = 0;
		}
		if (!block.isDescription()) {
			this.column += previousEnd - block.sourceStart + 1;
			FormatJavadocReference reference= block.reference;
			if (reference != null) {
				// format reference
				StringBuffer buffer = new StringBuffer();
				printJavadocBlockReference(buffer, reference, block);
			    addReplaceEdit(previousEnd+1, reference.sourceEnd, buffer.toString());
				previousEnd = reference.sourceEnd;
			}

			// Nothing else to do if the tag has no node
			if (maxNodes < 0)  {
				if (block.isInlined()) {
					// Need to print the closing brace
					StringBuffer buffer = new StringBuffer();
					if ((this.column+1) > maxColumn) {
						this.lastNumberOfNewLines++;
						this.line++;
				    	buffer.append(this.lineSeparator);
				    	this.column = 1;
				    	printIndentationIfNecessary(buffer);
			    		buffer.append(BLOCK_LINE_PREFIX);
				    	this.column = headerLine ? firstColumn-1 : firstColumn;
				    	if (indentationBuffer != null) {
							buffer.append(indentationBuffer);
							this.column += indentationBuffer.length();
				    	}
				    	headerLine = false;
				    	maxColumn--;
					}
					this.scanner.resetTo(previousEnd+1, block.sourceEnd+1);
					try {
	                    int token = this.scanner.getNextToken();
	                    while (token == TerminalTokens.TokenNameWHITESPACE || token == TerminalTokens.TokenNameMULTIPLY) {
	                    	token = this.scanner.getNextToken();
	                    }
	                    if (token == TerminalTokens.TokenNameRBRACE) {
		                    buffer.append(this.scanner.source, this.scanner.startPosition, this.scanner.currentPosition-this.scanner.startPosition);
					    	this.column += (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - this.scanner.startPosition;
	                    }
                    } catch (InvalidInputException e) {
						buffer.append('}');
                    }
                    addReplaceEdit(previousEnd+1, block.sourceEnd, buffer.toString());
				}
				return;
			}
		}
		
		// tag section: iterate through the blocks composing this tag but the last one
		for (int i=0; i<=maxNodes; i++) {
			FormatJavadocNode node = block.nodes[i];
			int nodeStart = node.sourceStart;
			
			// Print empty lines before the node
			int textLength = -1;
			int newLines;
			if (i == 0) {
				newLines = this.formatter.preferences.comment_insert_new_line_for_parameter && block.isParamTag() ? 1 : 0;
				if (nodeStart > (previousEnd+1)) {
					if (newLines == 0) {
						newLines = printJavadocBlockNodesNewLines(block, node, previousEnd);
					}
			   		printJavadocGapLines(previousEnd+1, nodeStart-1, newLines, clearBlankLines, false, null);
				} else {
					StringBuffer buffer = new StringBuffer();
					if (newLines > 0) {
						for (int j=0; j<newLines; j++) {
							printJavadocNewLine(buffer);
						}
						addInsertEdit(nodeStart, buffer.toString());
					}
				}
			} else {
				newLines = this.column > maxColumn ? 1 : 0;
				if (newLines < node.linesBefore) newLines = node.linesBefore;
				if (newLines == 0) {
					newLines = printJavadocBlockNodesNewLines(block, node, previousEnd);
				}
				if (newLines > 0 || nodeStart > (previousEnd+1)) {
		   			printJavadocGapLines(previousEnd+1, nodeStart-1, newLines, clearBlankLines, false, null);
				}
			}
			if (headerLine && newLines > 0) {
				headerLine = false;
				maxColumn--;
			}

			// Print node
			if (node.isText()) {
				FormatJavadocText text = (FormatJavadocText) node;
				if (text.isHtmlTag()) {
					if (text.isImmutableHtmlTag()) {
						// Immutable tags are not formatted
						if (textLength == -1) {
							textLength = getTextLength(block, text, false);
						}
						// Indent if new line was added
						int additionalLength = printJavadocHtmlImmutableTag(text, block, newLines > 0);
						this.column += textLength - additionalLength;
					} else {
						printJavadocHtmlTag(text, block, newLines>0);
					}
				} else {
					printJavadocText(text, block, newLines>0);
				}
			} else {
				if (newLines > 0 && indentationBuffer != null) {
					addInsertEdit(node.sourceStart, indentationBuffer.toString());
					this.column += indentationBuffer.length();
				}
				printJavadocBlock((FormatJavadocBlock)node);
			}

			// Print empty lines before the node
			previousEnd = node.sourceEnd;
		}
		this.lastNumberOfNewLines = 0;
	}

	private int printJavadocBlockNodesNewLines(FormatJavadocBlock block, FormatJavadocNode node, int previousEnd) {
	   	int maxColumn = this.formatter.preferences.comment_line_length+1;
    	int nodeStart = node.sourceStart;
 	    try {
			this.scanner.resetTo(nodeStart , node.sourceEnd);
	    	int token = this.scanner.getNextToken();
	    	int tokenLength = (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - this.scanner.startPosition;
	    	int newLines = 0;
	    	boolean newLine = false;
			boolean headerLine = block.isHeaderLine() && this.lastNumberOfNewLines == 0;
	    	if (node.isText()) {
	    		FormatJavadocText text = (FormatJavadocText)node;
    			if (text.isImmutableHtmlTag()) {
					int firstColumn = 1 + this.indentationLevel + BLOCK_LINE_PREFIX_LENGTH;
					if (headerLine) firstColumn++;
			    	if (nodeStart > (previousEnd+1)) {
			    		tokenLength++; // include space between nodes
			    	}
    				int col = this.column + tokenLength;
	    			while (!this.scanner.atEnd()) {
	    				token = this.scanner.getNextToken();
	    				switch (token) {
	    					case TerminalTokens.TokenNameWHITESPACE:
	    						if (CharOperation.indexOf('\n', this.scanner.source, this.scanner.startPosition, this.scanner.currentPosition) >= 0) {
	    							return newLines;
	    						}
	    						tokenLength = 1;
	    						break;
	    					case TerminalTokens.TokenNameMULTIPLY:
	    						if (newLine) {
	    							newLine = false;
	    							continue;
	    						}
	    						tokenLength = 1;
	    						break;
	    					default:
				    			tokenLength = (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - this.scanner.startPosition;
	    						break;
	    				}
	    				col += tokenLength;
			    		if (headerLine) { // special case when text is on the same line of the javadoc's header
			    			if ((col-1) > maxColumn)  {
								newLines++;
								col = firstColumn;
								firstColumn--;
								headerLine = false;
			    			}
			    		} else if (col > maxColumn) {
							newLines++;
							col = firstColumn;
						}
	    			}
	    			return newLines;
    			}
    			if (text.isHtmlTag()) {
	    			// read the html tag
	    			if (this.scanner.getNextToken() == TerminalTokens.TokenNameDIVIDE) {
	    				tokenLength++;
	    				this.scanner.getNextToken();
	    			}
	    			tokenLength += (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - this.scanner.startPosition;
	    			this.scanner.getNextToken(); // '>'
	    			tokenLength++;
	    		} else {
	    			while (true) {
	    				token = this.scanner.getNextToken();
	    				if (token == TerminalTokens.TokenNameWHITESPACE || token == TerminalTokens.TokenNameEOF) break;
		    			tokenLength += (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - this.scanner.startPosition;
	    			}
	    		}
	    	} else {
	    		FormatJavadocBlock inlinedBlock = (FormatJavadocBlock) node;
	    		tokenLength += inlinedBlock.tagEnd - inlinedBlock.sourceStart + 1;
	    	}
	    	if (nodeStart > (previousEnd+1)) {
	    		tokenLength++; // include space between nodes
	    	}
			if ((this.column + tokenLength) > maxColumn) {
	    		return 1;
	    	}
	    } catch (InvalidInputException iie) {
	    	// Assume length is one
	    	int tokenLength = 1;
	    	if (nodeStart > (previousEnd+1)) {
	    		tokenLength++; // include space between nodes
	    	}
			if ((this.column + tokenLength) > maxColumn) {
	    		return 1;
	    	}
	    }
	    return 0;
    }

	private void printJavadocBlockReference(StringBuffer buffer, FormatJavadocReference reference, FormatJavadocBlock block) {

		boolean indentRootTags = this.formatter.preferences.comment_indent_root_tags && !block.isInDescription();
		boolean indentParamTag = this.formatter.preferences.comment_indent_parameter_description && block.isInParamTag();
		boolean headerLine = block.isHeaderLine();
		StringBuffer tokensBuffer = new StringBuffer();
		
		// First we need to know what is the indentation
		int firstColumn = 1 + this.indentationLevel + BLOCK_LINE_PREFIX_LENGTH;
		if (headerLine) firstColumn++;
		StringBuffer indentationBuffer = null;
		if (indentRootTags) {
			int indentLevel = this.indentationLevel;
			int indentations = this.numberOfIndentations;
			this.numberOfIndentations += (BLOCK_LINE_PREFIX_LENGTH / this.indentationSize) + 1;
			this.indentationLevel = this.numberOfIndentations * this.indentationSize;
			int currentColumn = this.column;
			this.column = firstColumn;
			if (indentParamTag) {
				this.indentationLevel += this.indentationSize;
				this.numberOfIndentations++;
			}
			printIndentationIfNecessary(indentationBuffer = new StringBuffer());
			firstColumn = this.indentationLevel + 1;
			this.column = currentColumn;
			this.indentationLevel = indentLevel;
			this.numberOfIndentations = indentations;
		}

		// Scan the text token per token to compact it and size it the max line length
		int maxColumn = this.formatter.preferences.comment_line_length + 1;
		int previousToken = -1;
		this.scanner.resetTo(reference.sourceStart, reference.sourceEnd);
		while (!this.scanner.atEnd()) {
			int token;
			try {
				token = this.scanner.getNextToken();
				switch (token) {
					case TerminalTokens.TokenNameWHITESPACE:
						buffer.append(tokensBuffer);
						this.column += tokensBuffer.length();
						tokensBuffer.setLength(0);
						if (CharOperation.indexOf('\n', this.scanner.source, this.scanner.startPosition, this.scanner.getCurrentTokenEndPosition()) >= 0) {
							// consume line break
							loop: while (true) {
								token = this.scanner.getNextToken();
								switch (token) {
									case TerminalTokens.TokenNameWHITESPACE:
									case TerminalTokens.TokenNameMULTIPLY:
										previousToken = token; // will not insert space
										continue;
									default:
										break loop;
								}
							}
							break;
						}
						previousToken = token;
						continue;
					case TerminalTokens.TokenNameMULTIPLY:
						previousToken = token;
						continue;
				}
			} catch (InvalidInputException iie) {
				continue;
			}
			int tokenStart = this.scanner.getCurrentTokenStartPosition();
    		int tokenLength = (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - tokenStart;
    		int lastColumn = this.column + tokensBuffer.length() + tokenLength;
    		boolean insertSpace = previousToken == TerminalTokens.TokenNameWHITESPACE || previousToken == -1;
    		if (insertSpace) lastColumn++;
    		if (headerLine) {
    			// special case when text is on the same line of the javadoc's header
    			if (lastColumn > maxColumn)  {
    				lastColumn--; // new line gives an extra character
    			}
    		}
			if (lastColumn > maxColumn) {
		    	String tokensString = tokensBuffer.toString().trim();
		    	int indentLength = indentationBuffer==null ? 0 : indentationBuffer.length();
				if ((firstColumn-1+indentLength+tokensString.length()+tokenLength) > maxColumn) {
					// there won't be enough room even if we break the line before the buffered tokens
					// So add the buffered tokens now
					if (buffer.length() == 0) {
						buffer.append(' ');
						this.column++;
					}
					buffer.append(tokensString);
					this.column += tokensString.length();
					tokensBuffer.setLength(0);
				}
				// not enough space on the line
				this.lastNumberOfNewLines++;
				this.line++;
		    	buffer.append(this.lineSeparator);
		    	this.column = 1;
		    	printIndentationIfNecessary(buffer);
	    		buffer.append(BLOCK_LINE_PREFIX);
		    	this.column = headerLine ? firstColumn-1 : firstColumn;
		    	if (indentationBuffer != null) {
		    		buffer.append(indentationBuffer);
		    	}
		    	if (tokensBuffer.length() > 0) {
					buffer.append(tokensString);
					this.column += tokensString.length();
					tokensBuffer.setLength(0);
		    	}
				buffer.append(this.scanner.source, tokenStart, tokenLength);
				this.column += tokenLength;
				if (headerLine) {
					firstColumn--;
					headerLine = false;
				}
			} else {
				// append token to the line
				if (insertSpace) {
					tokensBuffer.append(' ');
				}
				tokensBuffer.append(this.scanner.source, tokenStart, tokenLength);
			}
			previousToken = token;
		}
		if (tokensBuffer.length() > 0) {
			buffer.append(tokensBuffer);
			this.column += tokensBuffer.length();
		}
    }

	private int getTextLength(FormatJavadocBlock block, FormatJavadocText text, boolean immutableEnd) {

		// Special case for immutable tags
		if (text.isImmutableHtmlTag()) {
			this.scanner.resetTo(text.sourceStart , text.sourceEnd);
			int textLength = 0;
			while (!this.scanner.atEnd()) {
				try {
	                int token = this.scanner.getNextToken();
	    			if (token == TerminalTokens.TokenNameWHITESPACE) {
						if (CharOperation.indexOf('\n', this.scanner.source, this.scanner.startPosition, this.scanner.currentPosition) >= 0) {
							if (immutableEnd) {
								textLength = 0;
							} else {
								return textLength;
							}
						}
	    			}
                } catch (InvalidInputException e) {
	                return textLength;
                }
    			textLength += (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - this.scanner.startPosition;
			}
			return textLength;
		}

		// Simple for one line tags
	    if (block.isOneLineTag()) {
	    	return text.sourceEnd - text.sourceStart + 1;
	    }
	    
	    // Find last line
    	int startLine = Util.getLineNumber(text.sourceStart, this.lineEnds, 0, this.maxLines);
    	int endLine = startLine;
    	int previousEnd = -1;
    	for (int i=0; i<=text.separatorsPtr; i++) {
    		int end = (int) (text.separators[i] >>> 32);
    		endLine = Util.getLineNumber(end, this.lineEnds, endLine-1, this.maxLines);
    		if (endLine > startLine) {
    			return previousEnd - text.sourceStart + 1;
    		}
    		previousEnd = end;
    	}

    	// This was a one line text
		return text.sourceEnd - text.sourceStart + 1;
    }
	
	/*
	 * Print and formats a javadoc comments
	 */
	void printJavadocComment(int start, int end) {
		int lastIndentationLevel = this.indentationLevel;
		try {
			// parse the comment on the fly
			this.scanner.resetTo(start, end-1);
			if (! this.formatterCommentParser.parse(start, end-1)) {
				// problem occurred while parsing the javadoc, early abort formatting
				return;
			}
			
			FormatJavadoc javadoc = (FormatJavadoc) this.formatterCommentParser.docComment;
			
			// handle indentation
			if (this.indentationLevel != 0) {
				printIndentationIfNecessary();
			}
			
			// handle pending space if any
			if (this.pendingSpace) {
				addInsertEdit(start, " "); //$NON-NLS-1$
			}
			
			if (javadoc.blocks == null) {
				// no FormatJavadocTags in this this javadoc
				return;
			}
			
			// init properly
			this.needSpace = false;
			this.pendingSpace = false;
			int length = javadoc.blocks.length;
			
			// format empty lines between before the first block
			FormatJavadocBlock previousBlock = javadoc.blocks[0];
			this.lastNumberOfNewLines = 0;
			int currentLine = this.line;
			int firstBlockStart = previousBlock.sourceStart;
			printIndentationIfNecessary(null);
			this.column += JAVADOC_HEADER_LENGTH; // consider that the header is already scanned
	
			// If there are several blocks in the javadoc
			int index = 1;
			if (length > 1) {
				// format the description if any
				if (previousBlock.isDescription()) {
					printJavadocBlock(previousBlock);
					FormatJavadocBlock block = javadoc.blocks[index++];
					int newLines = this.formatter.preferences.comment_insert_empty_line_before_root_tags ? 2 : 1;
					printJavadocGapLines(previousBlock.sourceEnd+1, block.sourceStart-1, newLines, this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment, false, null);
					previousBlock = block;
				}
	
				// format all tags but the last one composing this comment
				while (index < length) {
					printJavadocBlock(previousBlock);
					FormatJavadocBlock block = javadoc.blocks[index++];
					printJavadocGapLines(previousBlock.sourceEnd+1, block.sourceStart-1, 1, this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment, false, null);
					previousBlock = block;
				}
			}
	
			// format the last block
			printJavadocBlock(previousBlock);
	
			// format the header and footer empty spaces
			int newLines = this.line > currentLine || javadoc.isMultiLine() ? 1 : 0;
			printJavadocGapLines(javadoc.textStart, firstBlockStart-1, newLines, this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment, false, null);
			printJavadocGapLines(previousBlock.sourceEnd+1, javadoc.textEnd, newLines, this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment, true, null);
		}
		finally {
			// reset the scanner
			this.scanner.resetTo(end, this.scannerEndPosition - 1);
			needSpace = false;
			this.indentationLevel = lastIndentationLevel;
			this.lastNumberOfNewLines = 0;
		}
	}
	
	/*
	 * prints the empty javadoc line between the 2 given positions.
	 * May insert new '*' before each new line
	 */
	private void printJavadocGapLines(int textStartPosition, int textEndPosition, int newLines, boolean clearBlankLines, boolean footer, StringBuffer output) {
		try {
			// If no lines to set in the gap then just insert a space if there's enough room to
			if (newLines == 0) {
				if (output == null) {
					addReplaceEdit(textStartPosition, textEndPosition,  " "); //$NON-NLS-1$
				} else {
					output.append(' ');
				}
				this.column++;
				return;
			}

			// if there's no enough room to replace text, then insert the gap
			if (textStartPosition > textEndPosition) {
				if (newLines > 0) {
					StringBuffer buffer = new StringBuffer();
					for (int i=0; i<newLines; i++) {
						buffer.append(this.lineSeparator);
						this.column = 1;
						printIndentationIfNecessary(buffer);
						if (footer) {
							buffer.append(' ');
							this.column++;
						} else {
							buffer.append(BLOCK_LINE_PREFIX);
							this.column += BLOCK_LINE_PREFIX_LENGTH;
						}
					}
					if (output == null) {
						addInsertEdit(textStartPosition, buffer.toString());
					} else {
						output.append(buffer);
					}
				}
				return;
			}

			// There's enough room and some lines to set...
			// Skip the text token per token to keep existing stars when possible
			this.scanner.resetTo(textStartPosition, textEndPosition);
			this.scanner.recordLineSeparator = true;
			this.scanner.linePtr = Util.getLineNumber(textStartPosition, this.lineEnds, 0, this.maxLines) - 2;
			int linePtr = this.scanner.linePtr;
			int lineCount = 0;
			int start = textStartPosition;
			while (!this.scanner.atEnd()) {
				switch (this.scanner.getNextToken()) {
					case TerminalTokens.TokenNameMULTIPLY:
						// we just need to replace each lines between '*' with the javadoc formatted ones
						int linesGap = this.scanner.linePtr - linePtr;
						if (linesGap > 0) {
							StringBuffer buffer = new StringBuffer();
							if (lineCount > 0) {
								// TODO (eric) https://bugs.eclipse.org/bugs/show_bug.cgi?id=49619
								buffer.append( ' ');
							}
							for (int i = 0; i < linesGap ; i++) {
								if (clearBlankLines && lineCount >= newLines) {
									// leave as the required new lines have been inserted
									// so remove any remaining blanks and leave
									if (textEndPosition >= start) {
										if (output == null) {
											addReplaceEdit(start, textEndPosition, buffer.toString());
										} else {
											output.append(buffer);
										}
									}
									return;
								}
								buffer.append(this.lineSeparator);
								this.column = 1;
								printIndentationIfNecessary(buffer);
								if (i == (linesGap-1)) {
									buffer.append(' ');
									this.column++;
								} else {
									buffer.append(BLOCK_LINE_PREFIX);
									this.column += BLOCK_LINE_PREFIX_LENGTH;
								}
								lineCount++;
							}
							int currentTokenStartPosition = this.scanner.getCurrentTokenStartPosition();
							int tokenLength = this.scanner.currentPosition - currentTokenStartPosition;
							if (output == null) {
								addReplaceEdit(start, currentTokenStartPosition-1, buffer.toString());
							} else {
								output.append(buffer);
								output.append(this.scanner.source, currentTokenStartPosition, tokenLength);
							}
							this.column += tokenLength;
							if (footer && clearBlankLines && lineCount == newLines) {
								if (textEndPosition >= currentTokenStartPosition) {
									if (output == null) {
										addDeleteEdit(currentTokenStartPosition, textEndPosition);
									}
								}
								return;
							}
						}
						// next start is just after the current token
						start = this.scanner.currentPosition;
						linePtr = this.scanner.linePtr;
						break;
				}
			}

			// Format the last whitespaces
			if (lineCount < newLines) {
				// Insert new lines as not enough was encountered while scanning the whitespaces
				StringBuffer buffer = new StringBuffer();
				if (lineCount > 0) {
					// TODO (eric) https://bugs.eclipse.org/bugs/show_bug.cgi?id=49619
					buffer.append( ' ');
				}
				for (int i = lineCount; i < newLines-1; i++) {
					printJavadocNewLine(buffer);
				}
				buffer.append(this.lineSeparator);
				this.column = 1;
				printIndentationIfNecessary(buffer);
				if (footer) {
					buffer.append(' ');
					this.column++;
				} else {
					buffer.append(BLOCK_LINE_PREFIX);
					this.column += BLOCK_LINE_PREFIX_LENGTH;
				}
				if (output == null) {
					if (textEndPosition >= start) {
						addReplaceEdit(start, textEndPosition, buffer.toString());
					} else {
						addInsertEdit(textEndPosition+1, buffer.toString());
					}
				} else {
					output.append(buffer);
				}
			} else {
				// Replace all remaining whitespaces by a single space
				if (textEndPosition >= start) {
					StringBuffer buffer = new StringBuffer();
					if (this.scanner.linePtr > linePtr) {
						if (lineCount > 0) {
							// TODO (eric) https://bugs.eclipse.org/bugs/show_bug.cgi?id=49619
							buffer.append( ' ');
						}
						buffer.append(this.lineSeparator);
						this.column = 1;
						printIndentationIfNecessary(buffer);
					}
					buffer.append(' ');
					if (output == null) {
						addReplaceEdit(start, textEndPosition, buffer.toString());
					} else {
						output.append(buffer);
					}
				}
				this.column++;
			}
		}		
		catch (InvalidInputException iie) {
			// there's nothing to do if this exception happens
		}
		finally {
			this.scanner.recordLineSeparator = false;
			needSpace = false;
			this.scanner.resetTo(textEndPosition+1, this.scannerEndPosition - 1);
			this.lastNumberOfNewLines += newLines;
			this.line += newLines;
		}
	}

	private int printJavadocHtmlImmutableTag(FormatJavadocText text, FormatJavadocBlock block, boolean textOnNewLine) {

		int additionalLength = 0;
		try {
			// Indent if necessary
			if (textOnNewLine) {
				int col = this.column;
				StringBuffer buffer = new StringBuffer();
				int textEnd = text.separatorsPtr == -1 ? text.sourceEnd : (int) (text.separators[0] >>> 32);
				printJavadocTextLine(buffer, text.sourceStart, textEnd, block, true, true, true);
				addReplaceEdit(text.sourceStart, textEnd, buffer.toString());
				additionalLength = this.column - col;
			}
	
			// Iterate on text line separators
			int lineNumber = text.lineStart;
			this.scanner.tokenizeWhiteSpace = false;
			StringBuffer buffer = null;
			for (int idx=1, max=text.separatorsPtr; idx<max ; idx++) {
				int start = (int) text.separators[idx];
				int lineStart = Util.getLineNumber(start, lineEnds, lineNumber, this.maxLines);
				if (buffer == null) {
					buffer = new StringBuffer();
					this.column = 1;
					printIndentationIfNecessary(buffer);
					buffer.append(BLOCK_LINE_PREFIX);
					this.column += BLOCK_LINE_PREFIX_LENGTH;
				}
				while (lineNumber < lineStart) {
					int end = this.lineEnds[lineNumber-1];
					this.scanner.resetTo(end, start);
					int token = this.scanner.getNextToken();
					switch (token) {
						case TerminalTokens.TokenNameMULTIPLY:
						case TerminalTokens.TokenNameMULTIPLY_EQUAL:
							break;
						default:
							return 0;
					}
					if (this.scanner.currentCharacter == ' ') {
						this.scanner.getNextChar();
					}
					addReplaceEdit(end+1, this.scanner.getCurrentTokenEndPosition(), buffer.toString());
					lineNumber++;
				}
			}
		}
		catch (InvalidInputException iie) {
			// leave
		}
		finally {
			// Reset
			needSpace = false;
			this.scanner.tokenizeWhiteSpace = true;
			this.scanner.resetTo(text.sourceEnd+1, this.scannerEndPosition - 1);
		}
		return text.separatorsPtr<2 ? additionalLength : 0;
	}

	private int printJavadocHtmlTag(FormatJavadocText text, FormatJavadocBlock block, boolean textOnNewLine) {

		// Compute indentation if necessary
		boolean clearBlankLines = this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment;
		boolean headerLine = block.isHeaderLine() && this.lastNumberOfNewLines == 0;
		int firstColumn = 1 + this.indentationLevel + BLOCK_LINE_PREFIX_LENGTH;
		if (headerLine) firstColumn++;

		// Local variables init
		int textStart = text.sourceStart;
		int nextStart = textStart;
		int startLine = Util.getLineNumber(textStart, this.lineEnds, 0, this.maxLines);
	    int htmlTagID = text.getHtmlTagID();
	    StringBuffer buffer = new StringBuffer();

	    // New line will be added before next node
	    int max = text.separatorsPtr;
		int linesAfter = 0;
		int previousEnd = -1;
	    boolean isHtmlBreakTag = htmlTagID == JAVADOC_SINGLE_BREAK_TAG_ID;
		boolean isHtmlSeparatorTag = htmlTagID == JAVADOC_SEPARATOR_TAGS_ID;
		if (!isHtmlBreakTag) {

			// Iterate on text line separators
			boolean isCode = htmlTagID == JAVADOC_CODE_TAGS_ID;
			for (int idx=0, ptr=0; idx<=max || (text.htmlNodesPtr != -1 && ptr <= text.htmlNodesPtr); idx++) {

				// append text to buffer realigning with the line length
				int end = (idx > max) ? text.sourceEnd : (int) (text.separators[idx] >>> 32);
				int nodeKind = 0; // text break
				if (text.htmlNodesPtr >= 0 && ptr <= text.htmlNodesPtr && end > text.htmlNodes[ptr].sourceStart) {
					FormatJavadocNode node = text.htmlNodes[ptr];
					FormatJavadocText htmlTag = node.isText() ? (FormatJavadocText) node : null;
					int newLines = htmlTag == null ? 0 : htmlTag.linesBefore;
					if (linesAfter > newLines) {
						newLines = linesAfter;
						if (newLines > 1 && clearBlankLines) newLines = 1;
					}
					if (textStart < previousEnd) {
						addReplaceEdit(textStart, previousEnd, buffer.toString());
					}
					boolean immutable = htmlTag == null ? false : htmlTag.isImmutableHtmlTag();
					boolean overEndLine = false;
					if (immutable) {
						overEndLine = (this.column + getTextLength(block, htmlTag, false)) > (this.formatter.preferences.comment_line_length+1);
						if (overEndLine) {
							if (newLines < 1) newLines = 1;
						}
					}
					if (newLines == 0) {
						newLines = printJavadocBlockNodesNewLines(block, node, previousEnd);
					}
					if (newLines > 0 || (idx > 1 && (previousEnd+1) <= (nextStart-1))) {
						printJavadocGapLines(previousEnd+1, node.sourceStart-1, newLines, clearBlankLines, false, null);
					}
					if (newLines > 0) textOnNewLine = true;
					buffer = new StringBuffer();
					if (node.isText()) {
						if (immutable) {
							// do not change immutable tags, just increment column
							int additionalLength = printJavadocHtmlImmutableTag(htmlTag, block, overEndLine);
							this.column += getTextLength(block, htmlTag, true) - additionalLength;
							linesAfter = 0;
						} else {
							linesAfter = printJavadocHtmlTag(htmlTag, block, textOnNewLine);
						}
						nodeKind = 1; // text
					} else {
						printJavadocBlock((FormatJavadocBlock)node);
						linesAfter = 0;
						nodeKind = 2; // block
					}
					textStart = node.sourceEnd+1;
					ptr++;
					if (idx > max)  {
						return linesAfter;
					}
				} else {
					if (idx > 0 && linesAfter > 0) {
						printJavadocGapLines(previousEnd+1, nextStart, linesAfter, clearBlankLines, false, buffer);
						textOnNewLine = true;
					}
					boolean needIndentation = textOnNewLine;
					if (idx > 0) {
						if (!needIndentation && text.isTextAfterHtmlSeparatorTag(idx-1)) {
							needIndentation = true;
						}
					}
					boolean firstText = idx==1;
					if (idx > 1 && (previousEnd+1) > (nextStart-1)) {
						firstText = true;
					}
					printJavadocTextLine(buffer, nextStart, end, block, firstText, needIndentation, idx==0/* opening html tag?*/ || text.htmlIndexes[idx-1] != -1);
					linesAfter = 0;
				    if (idx==0) {
				    	if (isHtmlSeparatorTag) {
					    	linesAfter = 1;
					    }
					} else if (text.htmlIndexes[idx-1] == JAVADOC_SINGLE_BREAK_TAG_ID) {
				    	linesAfter = 1;
				    }
				}

				// Replace with current buffer if there are several empty lines between text lines
				nextStart = (int) text.separators[idx];
				int endLine = Util.getLineNumber(end, this.lineEnds, startLine-1, this.maxLines);
				startLine = Util.getLineNumber(nextStart, this.lineEnds, endLine-1, this.maxLines);
				int linesGap = startLine - endLine;
				if (linesGap > 0) {
					if (clearBlankLines && linesGap > 1) {
						// keep previously computed lines after
					} else {
						if (idx==0 || (idx==max && ((text.htmlIndexes[max] & JAVADOC_TAGS_ID_MASK) == htmlTagID)) || (idx < max && nodeKind==1 && (text.htmlIndexes[idx-1] & JAVADOC_TAGS_ID_MASK) != JAVADOC_IMMUTABLE_TAGS_ID)) {
							if (linesAfter < linesGap) {
								linesAfter = linesGap;
							}
						}
					}
				}
				textOnNewLine = linesAfter > 0;

				// print <pre> tag
				if (isCode) {
	    			int codeEnd = (int) (text.separators[max] >>> 32);
	    			if (codeEnd > end) {
	    				if (this.formatter.preferences.comment_format_source) {
							if (textStart < end) addReplaceEdit(textStart, end, buffer.toString());
			    			printJavadocGapLines(end+1, nextStart-1, 1, false/* never clear blank lines inside <pre> tag*/, false, null);
							printCodeSnippet(nextStart, codeEnd);
							nextStart = (int) text.separators[max];
		    				printJavadocGapLines(codeEnd+1, nextStart-1, 1, false/* never clear blank lines inside <pre> tag*/, false, null);
		    				return 2;
	    				}
	    			} else {
						nextStart = (int) text.separators[max];
						if ((nextStart-1) > (end+1)) {
							int line1 = Util.getLineNumber(end+1, this.lineEnds, startLine-1, this.maxLines);
							int line2 = Util.getLineNumber(nextStart-1, this.lineEnds, line1-1, this.maxLines);
		    				int gapLines = line2-line1-1;
							printJavadocGapLines(end+1, nextStart-1, gapLines, false/* never clear blank lines inside <pre> tag*/, false, null);
							if (gapLines > 0) textOnNewLine = true;
						}
	    			}
					return 1;
				}

				// store previous end
				previousEnd = end;
			}
	    }

		// Insert last gap
	    boolean closingTag = isHtmlBreakTag || (text.htmlIndexes != null && (text.htmlIndexes[max] & JAVADOC_TAGS_ID_MASK) == htmlTagID);
		if (previousEnd != -1) {
		    if (max > 0 && isHtmlSeparatorTag && closingTag) {
				if (linesAfter == 0) linesAfter = 1;
			}
			if (linesAfter > 0) {
				printJavadocGapLines(previousEnd+1, nextStart-1, linesAfter, clearBlankLines, false, buffer);
				textOnNewLine = true;
			}
		}

	    // Print closing tag
		boolean needIndentation = textOnNewLine;
		if (!needIndentation && !isHtmlBreakTag && text.htmlIndexes != null && text.isTextAfterHtmlSeparatorTag(max)) {
			needIndentation = true;
		}
		printJavadocTextLine(buffer, nextStart, text.sourceEnd, block, max<=0 /*not the first text*/, needIndentation, closingTag/* closing html tag*/);
		if (textStart < text.sourceEnd) {
			addReplaceEdit(textStart, text.sourceEnd, buffer.toString());
		}

		// Reset
		needSpace = false;
		this.scanner.resetTo(text.sourceEnd+1, this.scannerEndPosition - 1);

		// Return the new lines to insert after
	    if (max > 0 && isHtmlSeparatorTag) {
			return 1;
		}
	    return 0;
    }

	private StringBuffer printJavadocIndentationBuffer(FormatJavadocBlock block, int firstColumn) {
	    boolean indentRootTags = this.formatter.preferences.comment_indent_root_tags && !block.isInDescription();
		boolean indentParamTag = this.formatter.preferences.comment_indent_parameter_description && block.isInParamTag();
		StringBuffer indentationBuffer = null;
		if (indentRootTags) {
			int indentLevel = this.indentationLevel;
			int indentations = this.numberOfIndentations;
			this.numberOfIndentations += (BLOCK_LINE_PREFIX_LENGTH / this.indentationSize) + 1;
			this.indentationLevel = this.numberOfIndentations * this.indentationSize;
			int currentColumn = this.column;
			this.column = firstColumn;
			if (indentParamTag) {
				this.indentationLevel += this.indentationSize;
				this.numberOfIndentations++;
			}
			printIndentationIfNecessary(indentationBuffer = new StringBuffer());
			this.column = currentColumn;
			this.indentationLevel = indentLevel;
			this.numberOfIndentations = indentations;
		}
	    return indentationBuffer;
    }

	private void printJavadocNewLine(StringBuffer buffer) {
	    buffer.append(this.lineSeparator);
	    this.column = 1;
	    printIndentationIfNecessary(buffer);
	    buffer.append(BLOCK_LINE_PREFIX);
	    this.column += BLOCK_LINE_PREFIX_LENGTH;
	    this.line++;
	    this.lastNumberOfNewLines++;
    }

	private void printJavadocText(FormatJavadocText text, FormatJavadocBlock block, boolean textOnNewLine) {

		boolean clearBlankLines = this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment;
		StringBuffer buffer = new StringBuffer();
		int textStart = text.sourceStart;
		int nextStart = textStart;
		int startLine = Util.getLineNumber(textStart, this.lineEnds, 0, this.maxLines);

		// Iterate on text line separators
		for (int idx=0, max=text.separatorsPtr; idx<=max ; idx++) {

			// append text to buffer realigning with the line length
			int end = (int) (text.separators[idx] >>> 32);
			boolean needIndentation = buffer.length() == 0 && textOnNewLine;
			if (idx > 0) {
				if (!needIndentation && text.isTextAfterHtmlSeparatorTag(idx-1)) {
					needIndentation = true;
				}
			}
			printJavadocTextLine(buffer, nextStart, end, block, idx==0 /*first text?*/, needIndentation, false /*not an html tag*/);

			// Replace with current buffer if there are several empty lines between text lines
			nextStart = (int) text.separators[idx];
			if (!clearBlankLines) {
				int endLine = Util.getLineNumber(end, this.lineEnds, startLine-1, this.maxLines);
				startLine = Util.getLineNumber(nextStart, this.lineEnds, endLine-1, this.maxLines);
				if (startLine > (endLine+1)) {
					addReplaceEdit(textStart, end, buffer.toString());
					textStart = nextStart;
					buffer.setLength(0);
					printJavadocGapLines(end+1, nextStart-1, startLine - endLine, this.formatter.preferences.comment_clear_blank_lines_in_javadoc_comment, false, null);
					textOnNewLine = true;
				}
			}
		}
		
		// Replace remaining line
		boolean needIndentation = buffer.length() == 0 && textOnNewLine;
		printJavadocTextLine(buffer, nextStart, text.sourceEnd, block, text.separatorsPtr==-1 /* first text?*/, needIndentation, false /*not an html tag*/);
		// TODO (eric) Bring back following optimization
		// if (lastNewLines != this.lastNumberOfNewLines || (this.column - currentColumn) != (text.sourceEnd - text.sourceStart + 1)) {
			addReplaceEdit(textStart, text.sourceEnd, buffer.toString());
		// }

		// Reset
		needSpace = false;
		this.scanner.resetTo(text.sourceEnd+1, this.scannerEndPosition - 1);
	}

	/*
	 * Returns whether the text has been modified or not.
	 */
	private void printJavadocTextLine(StringBuffer buffer, int textStart, int textEnd, FormatJavadocBlock block, boolean firstText, boolean needIndentation, boolean isHtmlTag) {

		boolean indentRootTags = this.formatter.preferences.comment_indent_root_tags && !block.isInDescription();
		boolean indentParamTag = this.formatter.preferences.comment_indent_parameter_description && block.isInParamTag();
		boolean headerLine = block.isHeaderLine() && this.lastNumberOfNewLines == 0;
		
		// First we need to know what is the indentation
		StringBuffer tokensBuffer = new StringBuffer();
		int firstColumn = 1 + this.indentationLevel + BLOCK_LINE_PREFIX_LENGTH;
		if (headerLine) firstColumn++;
		StringBuffer indentationBuffer = null;
		if (indentRootTags) {
			int indentLevel = this.indentationLevel;
			int indentations = this.numberOfIndentations;
			this.numberOfIndentations += (BLOCK_LINE_PREFIX_LENGTH / this.indentationSize) + 1;
			this.indentationLevel = this.numberOfIndentations * this.indentationSize;
			int currentColumn = this.column;
			this.column = firstColumn;
			if (indentParamTag) {
				this.indentationLevel += this.indentationSize;
				this.numberOfIndentations++;
			}
			printIndentationIfNecessary(indentationBuffer = new StringBuffer());
			if (needIndentation) {
				this.column = firstColumn;
				printIndentationIfNecessary(buffer);
			}
			firstColumn = this.indentationLevel + 1;
			this.column = currentColumn < firstColumn ? firstColumn : currentColumn;
			this.indentationLevel = indentLevel;
			this.numberOfIndentations = indentations;
		} else if (this.column < firstColumn) {
			this.column = firstColumn;
		}

		// Scan the text token per token to compact it and size it the max line length
		int maxColumn = this.formatter.preferences.comment_line_length + 1;
		try {
			this.scanner.resetTo(textStart, textEnd);
			this.scanner.skipComments = true;
			int previousToken = -1;
			boolean openedString  =false;
			boolean hasTokens = false;

			// Consume text token per token
    		while (!this.scanner.atEnd()) {
				int token;
				try {
					token = this.scanner.getNextToken();
				} catch (InvalidInputException iie) {
					boolean insertSpace = (previousToken == TerminalTokens.TokenNameWHITESPACE || (this.scanner.startPosition == textStart && this.column > firstColumn && !firstText)) && !isHtmlTag;
					String msg = iie.getMessage();
					if (msg == Scanner.INVALID_CHARACTER_CONSTANT) {
						if (insertSpace) {
							tokensBuffer.append(' ');
						}
						tokensBuffer.append('\'');
					} else if (msg == Scanner.INVALID_CHAR_IN_STRING) {
						if (openedString) {
							openedString = false;
						} else {
							if (insertSpace) {
								tokensBuffer.append(' ');
							}
							openedString = true;
						}
						tokensBuffer.append('"');
					} else {
						// skip failure
					}
					// Need to retrieve correct position
					this.scanner.resetTo(this.scanner.startPosition, textEnd);
					this.scanner.getNextChar();
					previousToken = 1;
					continue;
				}
				switch (token) {
					case TerminalTokens.TokenNameWHITESPACE:
						previousToken = token;
						buffer.append(tokensBuffer);
						this.column += tokensBuffer.length();
						tokensBuffer.setLength(0);
						hasTokens = true;
						continue;
					case TerminalTokens.TokenNameStringLiteral:
						if (this.scanner.currentPosition > this.scanner.eofPosition) {
							this.scanner.resetTo(this.scanner.startPosition, textEnd);
							this.scanner.getNextChar();
							token = 1;
						}
						break;
					case TerminalTokens.TokenNameCharacterLiteral:
						if (this.scanner.currentPosition > this.scanner.eofPosition) {
							this.scanner.resetTo(this.scanner.startPosition, textEnd);
							this.scanner.getNextChar();
							token = 1;
						}
						break;
				}
    			int tokenStart = this.scanner.getCurrentTokenStartPosition();
	    		int tokenLength = (this.scanner.atEnd() ? this.scanner.eofPosition : this.scanner.currentPosition) - tokenStart;
	    		boolean insertSpace = previousToken == TerminalTokens.TokenNameWHITESPACE || (tokenStart == textStart && this.column > firstColumn && !(firstText || isHtmlTag));
	    		int lastColumn = this.column + tokensBuffer.length() + tokenLength;
	    		if (insertSpace) lastColumn++;
	    		if (headerLine) {
	    			// special case when text is on the same line of the javadoc's header
	    			if (lastColumn > maxColumn)  {
	    				lastColumn--; // new line gives an extra character
	    				this.lastNumberOfNewLines++; // in case we leave just after
	    				this.line++;
	    			}
	    		}
				if (lastColumn > maxColumn) {
					// not enough space on the line
					boolean alreadyHasNewLine = !hasTokens && tokensBuffer.length() > 0 && firstText && !isHtmlTag && this.lastNumberOfNewLines > 0;
					if (!alreadyHasNewLine) {
						if (lastColumn == this.column) {
							this.lastNumberOfNewLines++;
							this.line++;
						}
				    	buffer.append(this.lineSeparator);
				    	this.column = 1;
				    	printIndentationIfNecessary(buffer);
			    		buffer.append(BLOCK_LINE_PREFIX);
				    	this.column = headerLine ? firstColumn-1 : firstColumn;
				    	if (indentationBuffer != null) {
				    		buffer.append(indentationBuffer);
				    	}
					}
			    	if (tokensBuffer.length() > 0) {
			    		String tokensString = tokensBuffer.toString().trim();
						buffer.append(tokensString);
						this.column += tokensString.length();
						tokensBuffer.setLength(0);
			    	}
					buffer.append(this.scanner.source, tokenStart, tokenLength);
					hasTokens = true;
					this.column += tokenLength;
					if (headerLine) {
						firstColumn--;
						headerLine = false;
					}
    			} else {
					// append token to the line
		    		if (insertSpace) {
		    			tokensBuffer.append(' ');
		    		}
					tokensBuffer.append(this.scanner.source, tokenStart, tokenLength);
    			}
				previousToken = token;
    		}
		}
		finally {
			this.scanner.skipComments = false;
			// Add remaining buffered tokens
			if (tokensBuffer.length() > 0) {
				buffer.append(tokensBuffer);
				this.column += tokensBuffer.length();
			}
		}
    }

	public void printModifiers(Annotation[] annotations, ASTVisitor visitor) {
		printModifiers(annotations, visitor, ICodeFormatterConstants.ANNOTATION_UNSPECIFIED);
	}
	
	public void printModifiers(Annotation[] annotations, ASTVisitor visitor, int annotationSourceKind) {
		try {
			int annotationsLength = annotations != null ? annotations.length : 0;
			int annotationsIndex = 0;
			boolean isFirstModifier = true;
			int currentTokenStartPosition = this.scanner.currentPosition;
			boolean hasComment = false;
			boolean hasModifiers = false;
			while ((this.currentToken = this.scanner.getNextToken()) != TerminalTokens.TokenNameEOF) {
				switch(this.currentToken) {
					case TerminalTokens.TokenNamepublic :
					case TerminalTokens.TokenNameprotected :
					case TerminalTokens.TokenNameprivate :
					case TerminalTokens.TokenNamestatic :
					case TerminalTokens.TokenNameabstract :
					case TerminalTokens.TokenNamefinal :
					case TerminalTokens.TokenNamenative :
					case TerminalTokens.TokenNamesynchronized :
					case TerminalTokens.TokenNametransient :
					case TerminalTokens.TokenNamevolatile :
					case TerminalTokens.TokenNamestrictfp :
						hasModifiers = true;
						print(this.scanner.currentPosition - this.scanner.startPosition, !isFirstModifier);
						isFirstModifier = false;
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameAT :
						hasModifiers = true;
						if (!isFirstModifier) {
							this.space();
						}
						this.scanner.resetTo(this.scanner.getCurrentTokenStartPosition(), this.scannerEndPosition - 1);
						if (annotationsIndex < annotationsLength) {
							annotations[annotationsIndex++].traverse(visitor, (BlockScope) null);
							// https://bugs.eclipse.org/bugs/show_bug.cgi?id=122247
							boolean shouldAddNewLine = false;
							switch (annotationSourceKind) {
								case ICodeFormatterConstants.ANNOTATION_ON_MEMBER :
									if (this.formatter.preferences.insert_new_line_after_annotation_on_member) {
										shouldAddNewLine = true;
									}
									break;
								case ICodeFormatterConstants.ANNOTATION_ON_PARAMETER :
									if (this.formatter.preferences.insert_new_line_after_annotation_on_parameter) {
										shouldAddNewLine = true;
									}
									break;
								case ICodeFormatterConstants.ANNOTATION_ON_LOCAL_VARIABLE :
									if (this.formatter.preferences.insert_new_line_after_annotation_on_local_variable) {
										shouldAddNewLine = true;
									}
									break;
								default:
									// do nothing when no annotation formatting option specified
							}
							if (shouldAddNewLine) {
								this.printNewLine();
							}
						} else {
							return;
						}
						isFirstModifier = false;
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameCOMMENT_BLOCK :
						printBlockComment(false);
						currentTokenStartPosition = this.scanner.currentPosition;
						hasComment = true;
						break;
					case TerminalTokens.TokenNameCOMMENT_JAVADOC :
						printBlockComment(true);
						currentTokenStartPosition = this.scanner.currentPosition;
						hasComment = true;
						break;
					case TerminalTokens.TokenNameCOMMENT_LINE :
						printLineComment();
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameWHITESPACE :
						addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						int count = 0;
						char[] whiteSpaces = this.scanner.getCurrentTokenSource();
						for (int i = 0, max = whiteSpaces.length; i < max; i++) {
							switch(whiteSpaces[i]) {
								case '\r' :
									if ((i + 1) < max) {
										if (whiteSpaces[i + 1] == '\n') {
											i++;
										}
									}
									count++;
									break;
								case '\n' :
									count++;
							}
						}
						if (count >= 1 && hasComment) {
							printNewLine();
						}
						currentTokenStartPosition = this.scanner.currentPosition;
						hasComment = false;
						break;
					default:
						if (hasModifiers) {
							this.space();
						}
						// step back one token
						this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
						return;					
				}
			}
		} catch (InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}
	
	public void printNewLine() {
		this.printNewLine(this.scanner.getCurrentTokenEndPosition() + 1);
	}

	public void printNewLine(int insertPosition) {
		if (this.nlsTagCounter > 0) {
			return;
		}
		if (lastNumberOfNewLines >= 1) {
			column = 1; // ensure that the scribe is at the beginning of a new line
			return;
		}
		addInsertEdit(insertPosition, this.lineSeparator);
		line++;
		lastNumberOfNewLines = 1;
		column = 1;
		needSpace = false;
		this.pendingSpace = false;
	}

	public void printNextToken(int expectedTokenType){
		printNextToken(expectedTokenType, false);
	}

	public void printNextToken(int expectedTokenType, boolean considerSpaceIfAny){
		printComment(CodeFormatter.K_UNKNOWN);
		try {
			this.currentToken = this.scanner.getNextToken();
			if (expectedTokenType != this.currentToken) {
				throw new AbortFormatting("unexpected token type, expecting:"+expectedTokenType+", actual:"+this.currentToken);//$NON-NLS-1$//$NON-NLS-2$
			}
			print(this.scanner.currentPosition - this.scanner.startPosition, considerSpaceIfAny);
		} catch (InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}

	public void printNextToken(int[] expectedTokenTypes) {
		printNextToken(expectedTokenTypes, false);
	}

	public void printNextToken(int[] expectedTokenTypes, boolean considerSpaceIfAny){
		printComment(CodeFormatter.K_UNKNOWN);
		try {
			this.currentToken = this.scanner.getNextToken();
			if (Arrays.binarySearch(expectedTokenTypes, this.currentToken) < 0) {
				StringBuffer expectations = new StringBuffer(5);
				for (int i = 0; i < expectedTokenTypes.length; i++){
					if (i > 0) {
						expectations.append(',');
					}
					expectations.append(expectedTokenTypes[i]);
				}				
				throw new AbortFormatting("unexpected token type, expecting:["+expectations.toString()+"], actual:"+this.currentToken);//$NON-NLS-1$//$NON-NLS-2$
			}
			print(this.scanner.currentPosition - this.scanner.startPosition, considerSpaceIfAny);
		} catch (InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}

	public void printArrayQualifiedReference(int numberOfTokens, int sourceEnd) {
		int currentTokenStartPosition = this.scanner.currentPosition;
		int numberOfIdentifiers = 0;
		try {
			do {
				printComment(CodeFormatter.K_UNKNOWN);
				switch(this.currentToken = this.scanner.getNextToken()) {
					case TerminalTokens.TokenNameEOF :
						return;
					case TerminalTokens.TokenNameWHITESPACE :
						addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameCOMMENT_BLOCK :
					case TerminalTokens.TokenNameCOMMENT_JAVADOC :
						printBlockComment(false);
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameCOMMENT_LINE :
						printLineComment();
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameIdentifier :
						print(this.scanner.currentPosition - this.scanner.startPosition, false);
						currentTokenStartPosition = this.scanner.currentPosition;
						if (++ numberOfIdentifiers == numberOfTokens) {
							this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
							return;
						}
						break;						
					case TerminalTokens.TokenNameDOT :
						print(this.scanner.currentPosition - this.scanner.startPosition, false);
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameRPAREN:
						currentTokenStartPosition = this.scanner.startPosition;
						// fall through default case...
					default:
						this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
						return;
				}
			} while (this.scanner.currentPosition <= sourceEnd);
		} catch(InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}

	public void printQualifiedReference(int sourceEnd, boolean expectParenthesis) {
		int currentTokenStartPosition = this.scanner.currentPosition;
		try {
			do {
				printComment(CodeFormatter.K_UNKNOWN);
				switch(this.currentToken = this.scanner.getNextToken()) {
					case TerminalTokens.TokenNameEOF :
						return;
					case TerminalTokens.TokenNameWHITESPACE :
						addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameCOMMENT_BLOCK :
					case TerminalTokens.TokenNameCOMMENT_JAVADOC :
						printBlockComment(false);
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameCOMMENT_LINE :
						printLineComment();
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameIdentifier :
					case TerminalTokens.TokenNameDOT :
						print(this.scanner.currentPosition - this.scanner.startPosition, false);
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					case TerminalTokens.TokenNameRPAREN:
						if (expectParenthesis) {
							currentTokenStartPosition = this.scanner.startPosition;
						}
						// fall through default case...
					default:
						this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
						return;
				}
			} while (this.scanner.currentPosition <= sourceEnd);
		} catch(InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}

	private void printRule(StringBuffer stringBuffer) {
		for (int i = 0; i < this.pageWidth; i++){
			if ((i % this.tabLength) == 0) { 
				stringBuffer.append('+');
			} else {
				stringBuffer.append('-');
			}
		}
		stringBuffer.append(this.lineSeparator);
		
		for (int i = 0; i < (pageWidth / tabLength); i++) {
			stringBuffer.append(i);
			stringBuffer.append('\t');
		}			
	}

	public void printTrailingComment(int numberOfNewLinesToInsert) {
		try {
			// if we have a space between two tokens we ensure it will be dumped in the formatted string
			int currentTokenStartPosition = this.scanner.currentPosition;
			boolean hasWhitespaces = false;
			boolean hasLineComment = false;
			while ((this.currentToken = this.scanner.getNextToken()) != TerminalTokens.TokenNameEOF) {
				switch(this.currentToken) {
					case TerminalTokens.TokenNameWHITESPACE :
						int count = 0;
						char[] whiteSpaces = this.scanner.getCurrentTokenSource();
						for (int i = 0, max = whiteSpaces.length; i < max; i++) {
							switch(whiteSpaces[i]) {
								case '\r' :
									if ((i + 1) < max) {
										if (whiteSpaces[i + 1] == '\n') {
											i++;
										}
									}
									count++;
									break;
								case '\n' :
									count++;
							}
						}
						if (hasLineComment) {
							if (count >= 1) {
								currentTokenStartPosition = this.scanner.getCurrentTokenStartPosition();
								this.preserveEmptyLines(numberOfNewLinesToInsert, currentTokenStartPosition);
								addDeleteEdit(currentTokenStartPosition, this.scanner.getCurrentTokenEndPosition());
								this.scanner.resetTo(this.scanner.currentPosition, this.scannerEndPosition - 1);
								return;
							}
							this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
							return;
						} else if (count > 1) {
							this.printEmptyLines(numberOfNewLinesToInsert, this.scanner.getCurrentTokenStartPosition());
							this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
							return;
						} else {
							hasWhitespaces = true;
							currentTokenStartPosition = this.scanner.currentPosition;						
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						}
						break;
					case TerminalTokens.TokenNameCOMMENT_LINE :
						if (hasWhitespaces) {
							space();
						}
						printLineComment();
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = true;
						break;
					case TerminalTokens.TokenNameCOMMENT_BLOCK :
						if (hasWhitespaces) {
							space();
						}
						printBlockComment(false);
						currentTokenStartPosition = this.scanner.currentPosition;
						break;
					default :
						// step back one token
						this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
						return;
				}
			}
		} catch (InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}
	public void printTrailingComment() {
		try {
			// if we have a space between two tokens we ensure it will be dumped in the formatted string
			int currentTokenStartPosition = this.scanner.currentPosition;
			boolean hasWhitespaces = false;
			boolean hasComment = false;
			boolean hasLineComment = false;
			while ((this.currentToken = this.scanner.getNextToken()) != TerminalTokens.TokenNameEOF) {
				switch(this.currentToken) {
					case TerminalTokens.TokenNameWHITESPACE :
						int count = 0;
						char[] whiteSpaces = this.scanner.getCurrentTokenSource();
						for (int i = 0, max = whiteSpaces.length; i < max; i++) {
							switch(whiteSpaces[i]) {
								case '\r' :
									if ((i + 1) < max) {
										if (whiteSpaces[i + 1] == '\n') {
											i++;
										}
									}
									count++;
									break;
								case '\n' :
									count++;
							}
						}
						if (hasLineComment) {
							if (count >= 1) {
								currentTokenStartPosition = this.scanner.getCurrentTokenStartPosition();
								this.preserveEmptyLines(count, currentTokenStartPosition);
								addDeleteEdit(currentTokenStartPosition, this.scanner.getCurrentTokenEndPosition());
								this.scanner.resetTo(this.scanner.currentPosition, this.scannerEndPosition - 1);
								return;
							}
							this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
							return;
						} else if (count >= 1) {
							if (hasComment) {
								this.printNewLine(this.scanner.getCurrentTokenStartPosition());
							}
							this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
							return;
						} else {
							hasWhitespaces = true;
							currentTokenStartPosition = this.scanner.currentPosition;
							addDeleteEdit(this.scanner.getCurrentTokenStartPosition(), this.scanner.getCurrentTokenEndPosition());
						}
						break;
					case TerminalTokens.TokenNameCOMMENT_LINE :
						if (hasWhitespaces) {
							space();
						}
						printLineComment();
						currentTokenStartPosition = this.scanner.currentPosition;
						hasLineComment = true;
						break;
					case TerminalTokens.TokenNameCOMMENT_BLOCK :
						if (hasWhitespaces) {
							space();
						}
						printBlockComment(false);
						currentTokenStartPosition = this.scanner.currentPosition;
						hasComment = true;
						break;
					default :
						// step back one token
						this.scanner.resetTo(currentTokenStartPosition, this.scannerEndPosition - 1);
						return;
				}
			}
		} catch (InvalidInputException e) {
			throw new AbortFormatting(e);
		}
	}

	void redoAlignment(AlignmentException e){
		if (e.relativeDepth > 0) { // if exception targets a distinct context
			e.relativeDepth--; // record fact that current context got traversed
			this.currentAlignment = this.currentAlignment.enclosing; // pop currentLocation
			throw e; // rethrow
		} 
		// reset scribe/scanner to restart at this given location
		this.resetAt(this.currentAlignment.location);
		this.scanner.resetTo(this.currentAlignment.location.inputOffset, this.scanner.eofPosition);
		// clean alignment chunkKind so it will think it is a new chunk again
		this.currentAlignment.chunkKind = 0;
	}

	void redoMemberAlignment(AlignmentException e){
		// reset scribe/scanner to restart at this given location
		this.resetAt(this.memberAlignment.location);
		this.scanner.resetTo(this.memberAlignment.location.inputOffset, this.scanner.eofPosition);
		// clean alignment chunkKind so it will think it is a new chunk again
		this.memberAlignment.chunkKind = 0;
	}

	public void reset() {
		this.checkLineWrapping = true;
		this.line = 0;
		this.column = 1;
		this.editsIndex = 0;
		this.nlsTagCounter = 0;
	}
		
	private void resetAt(Location location) {
		this.line = location.outputLine;
		this.column = location.outputColumn;
		this.indentationLevel = location.outputIndentationLevel;
		this.numberOfIndentations = location.numberOfIndentations;
		this.lastNumberOfNewLines = location.lastNumberOfNewLines;
		this.needSpace = location.needSpace;
		this.pendingSpace = location.pendingSpace;
		this.editsIndex = location.editsIndex;
		this.nlsTagCounter = location.nlsTagCounter;
		if (this.editsIndex > 0) {
			this.edits[this.editsIndex - 1] = location.textEdit;
		}
		this.formatter.lastLocalDeclarationSourceStart = location.lastLocalDeclarationSourceStart;
	}

	private void resize() {
		System.arraycopy(this.edits, 0, (this.edits = new OptimizedReplaceEdit[this.editsIndex * 2]), 0, this.editsIndex);
	}

	void setIncludeComments(boolean on) {
		if (on) {
			this.formatComments |= CodeFormatter.F_INCLUDE_COMMENTS;
		} else {
			this.formatComments &= ~CodeFormatter.F_INCLUDE_COMMENTS;
		}
	}
	
	void setHeaderComment(int position) {
		this.headerEndPosition = position;
	}

	public void space() {
		if (!this.needSpace) return;
		this.lastNumberOfNewLines = 0;
		this.pendingSpace = true;
		this.column++;
		this.needSpace = false;		
	}

	public String toString() {
		StringBuffer stringBuffer = new StringBuffer();
		stringBuffer
			.append("(page width = " + this.pageWidth + ") - (tabChar = ");//$NON-NLS-1$//$NON-NLS-2$
		switch(this.tabChar) {
			case DefaultCodeFormatterOptions.TAB :
				 stringBuffer.append("TAB");//$NON-NLS-1$
				 break;
			case DefaultCodeFormatterOptions.SPACE :
				 stringBuffer.append("SPACE");//$NON-NLS-1$
				 break;
			default :
				 stringBuffer.append("MIXED");//$NON-NLS-1$
		}
		stringBuffer
			.append(") - (tabSize = " + this.tabLength + ")")//$NON-NLS-1$//$NON-NLS-2$
			.append(this.lineSeparator)
			.append("(line = " + this.line + ") - (column = " + this.column + ") - (identationLevel = " + this.indentationLevel + ")")	//$NON-NLS-1$	//$NON-NLS-2$	//$NON-NLS-3$	//$NON-NLS-4$
			.append(this.lineSeparator)
			.append("(needSpace = " + this.needSpace + ") - (lastNumberOfNewLines = " + this.lastNumberOfNewLines + ") - (checkLineWrapping = " + this.checkLineWrapping + ")")	//$NON-NLS-1$	//$NON-NLS-2$	//$NON-NLS-3$	//$NON-NLS-4$
			.append(this.lineSeparator)
			.append("==================================================================================")	//$NON-NLS-1$
			.append(this.lineSeparator);
		printRule(stringBuffer);
		return stringBuffer.toString();
	}
	
	public void unIndent() {
		this.indentationLevel -= this.indentationSize;
		this.numberOfIndentations--;
	}
}
