package pl.poznan.put.qjunit.view;

import java.lang.reflect.Field;

import org.eclipse.core.runtime.Preferences;
import org.eclipse.jdt.internal.junit.model.TestCaseElement;
import org.eclipse.jdt.internal.junit.ui.TestRunnerViewPart;
import org.eclipse.jdt.internal.junit.ui.TestViewer;
import org.eclipse.jdt.junit.JUnitCore;
import org.eclipse.jdt.junit.TestRunListener;
import org.eclipse.jdt.junit.model.ITestCaseElement;
import org.eclipse.jdt.junit.model.ITestRunSession;
import org.eclipse.jdt.junit.model.ITestSuiteElement;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.part.ViewPart;

import pl.poznan.put.qjunit.Activator;
import pl.poznan.put.qjunit.model.MutationElement;
import pl.poznan.put.qjunit.model.MutationOperator;
import pl.poznan.put.qjunit.model.MutationResult;
import pl.poznan.put.qjunit.model.MutationSession;
import pl.poznan.put.qjunit.model.TestCaseAdapter;

public class MutationsView extends ViewPart {

	public static final String ID = "pl.poznan.put.qjunit.view";
	
	public static final String SHOW_CONTENT = "show_content";
	public static final String CONTENT_DETAILS = "details";
	public static final String CONTENT_SUMMARY = "summary";
	public static final String CONTENT_OPERATORS = "operators";
	public static final String FILTER_DEAD = "filter_dead";
	
	public static class DeadMutantsFilter extends ViewerFilter {

		public boolean select(Viewer viewer, Object parentElement,
				Object element) {
			
			if (element instanceof MutationResult) {
				return MutationResult.OK.equals(((MutationResult) element).getResult());
			}
			
			if (element instanceof MutationElement) {
				MutationElement mutationElement = (MutationElement) element;
				return (mutationElement.getResults().length > 0) &&
					(mutationElement.getResultsAlive() > 0);
			}
			
			if (element instanceof ITestCaseElement) {
				ITestCaseElement tc = (ITestCaseElement) element;
				MutationSession session = Activator.getMutationModel().getMutationSession();
				element = session.getTestCaseAdapter(tc.getTestClassName(), tc.getTestMethodName());
			}
			
			if (element instanceof TestCaseAdapter) {
				TestCaseAdapter tca = ((TestCaseAdapter) element);
				return tca.getMutationsCount() > 0 && tca.getMutationsAlive() > 0;
			}
			
			return true;
		}
		
	}
	
	private ISelectionChangedListener junitViewListener;
	
	private TestRunListener testRunListener;
	
	private ViewerFilter deadMutantsFilter;
	
	private TreeViewer fViewer;
	private IAction showSummaryAction;
	private IAction showTestDetailsAction;
	private IAction showOperatorsAction;
	private IAction filterDeadMutantsAction;
	private IMemento fMemento;
	
	public MutationsView() {
		junitViewListener = new ISelectionChangedListener() {

			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection selection = (IStructuredSelection) event.getSelection();
				showDetails(selection);
			}
		};
		
		testRunListener = new TestRunListener() {

			private void updateInUIThread() {
				getSite().getShell().getDisplay().asyncExec(new Runnable() {
					public void run() {
						updateContents();
					}
				});
			}
			
			public void sessionFinished(ITestRunSession session) {
				updateInUIThread();
			}
			
			public void sessionStarted(ITestRunSession session) {
				updateInUIThread();
			}
			
		};
		
		deadMutantsFilter = new DeadMutantsFilter();
	}
	
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		
		if (memento == null) {
			memento = XMLMemento.createWriteRoot(ID);
		}
		
		fMemento = memento;
		
		readSettings();
	}
	
	

	public void saveState(IMemento memento) {
		super.saveState(memento);
		
		writeSettings();
		
		memento.putMemento(memento);
	}

	public void createPartControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout gl = new GridLayout(1, false);
		gl.marginWidth = 0;
		gl.marginHeight = 0;
		composite.setLayout(gl);
		
		fViewer = new TreeViewer(composite, SWT.V_SCROLL | SWT.SINGLE);
		Tree tree = fViewer.getTree();
		tree.setLayoutData(new GridData(GridData.FILL_BOTH));
		fViewer.setLabelProvider(new MutationsViewLabelProvider(fMemento));
		fViewer.setContentProvider(new MutationsViewContentProvider());
		
		TreeColumn tc = new TreeColumn(tree, SWT.LEFT);
		tc.setMoveable(true);
		tc.setResizable(true);
		tc.setText("Test Case");
		tc.setWidth(250);
		
		tc = new TreeColumn(tree, SWT.LEFT);
		tc.setMoveable(true);
		tc.setResizable(true);
		tc.setText("Alive");
		tree.setHeaderVisible(true);
		tc.setWidth(50);
		
		tc = new TreeColumn(tree, SWT.LEFT);
		tc.setMoveable(true);
		tc.setResizable(true);
		tc.setText("Name");
		tree.setHeaderVisible(true);
		tc.setWidth(50);
		
		fViewer.addDoubleClickListener(new IDoubleClickListener() {

			public void doubleClick(DoubleClickEvent event) {
				IStructuredSelection selection = (IStructuredSelection) event.getSelection();
				if (! selection.isEmpty()) {
					openSelection(selection.getFirstElement());
				}
			}
			
		});
		
		updateFilters();
		
		makeActions();
		hookMenuActions();
		
		JUnitCore.addTestRunListener(testRunListener);
		
		updateContents();
	}
	
	protected void updateFilters() {
		if (Boolean.valueOf(fMemento.getString(FILTER_DEAD)).booleanValue()) {
			fViewer.addFilter(deadMutantsFilter);
		} else {
			fViewer.removeFilter(deadMutantsFilter);
		}
	}

	protected void makeActions() {
		showSummaryAction = new Action("Mutations summary", IAction.AS_RADIO_BUTTON) {
			public void run() {
				fMemento.putString(SHOW_CONTENT, CONTENT_SUMMARY);
				showSummary();
			}
		};
		showSummaryAction.setToolTipText("Enables all mutations summary report");
		showSummaryAction.setChecked(CONTENT_SUMMARY.equals(fMemento.getString(SHOW_CONTENT)));
		
		showTestDetailsAction = new Action("Mutation details", IAction.AS_RADIO_BUTTON) {
			public void run() {
				fMemento.putString(SHOW_CONTENT, CONTENT_DETAILS);
				
				TreeViewer junit = getJUnitTreeViewer(true);
				showDetails((IStructuredSelection) junit.getSelection());
			}
		};
		showTestDetailsAction.setToolTipText("Enables mutation details for currently selected test");
		showTestDetailsAction.setChecked(CONTENT_DETAILS.equals(fMemento.getString(SHOW_CONTENT)));
		
		showOperatorsAction = new Action("Operators summary", IAction.AS_RADIO_BUTTON) {
			public void run() {
				fMemento.putString(SHOW_CONTENT, CONTENT_OPERATORS);
				
				showOperators();
			}
		};
		showOperatorsAction.setToolTipText("Switch to operators details");
		showOperatorsAction.setChecked(CONTENT_OPERATORS.equals(fMemento.getString(SHOW_CONTENT)));
		
		filterDeadMutantsAction = new Action("Failures only", IAction.AS_CHECK_BOX) {
			public void run() {
				fMemento.putString(FILTER_DEAD, Boolean.toString(filterDeadMutantsAction.isChecked()));
				updateFilters();
			}
		};
		filterDeadMutantsAction.setToolTipText("Filters out killer tests, leaving suspected ones.");
		filterDeadMutantsAction.setChecked(Boolean.valueOf(fMemento.getString(FILTER_DEAD)).booleanValue());
	}
	
	protected void hookMenuActions() {
		IMenuManager mgr = getViewSite().getActionBars().getMenuManager();
		mgr.add(showSummaryAction);
		mgr.add(showTestDetailsAction);
		mgr.add(showOperatorsAction);
		mgr.add(new Separator());
		mgr.add(filterDeadMutantsAction);
	}
	
	protected void showDetails(IStructuredSelection selection) {
		if (! CONTENT_DETAILS.equals(fMemento.getString(SHOW_CONTENT))) {
			return;
		}
		
		if (selection.isEmpty()) {
			fViewer.setInput(null);
		} else {
			fViewer.setInput(selection.toArray());
		}
		
		MutationSession session = Activator.getMutationModel().getMutationSession();
		if (session != null) {
			Object[] elements = selection.toArray();
			
			int count = 0;
			int alive = 0;
			
			for (int i = 0; i < elements.length; i++) {
				if (elements[i] instanceof TestCaseElement) {
					TestCaseElement tce = (TestCaseElement) elements[i];
					TestCaseAdapter adapter = session.getTestCaseAdapter(tce.getTestClassName(), tce.getTestMethodName());
					if (adapter != null) {
						count += adapter.getMutationsCount();
						alive += adapter.getMutationsAlive();
					}
				}
			}
			
			setContentDescription(alive + " out of total "+ count + " responses are alive");
		} else {
			setContentDescription("");
		}
	}
	
	protected void updateContents() {
		if (CONTENT_DETAILS.equals(fMemento.getString(SHOW_CONTENT))) {
			TreeViewer viewer = getJUnitTreeViewer(false);
			if (viewer != null)
			showDetails((IStructuredSelection) viewer.getSelection());
		} else if (CONTENT_OPERATORS.equals(fMemento.getString(SHOW_CONTENT))) {
			showOperators();
		} else {
			showSummary();
		}
	}
	
	protected void showOperators() {
		if (! CONTENT_OPERATORS.equals(fMemento.getString(SHOW_CONTENT))) {
			return;
		}
		
		MutationSession session = Activator.getMutationModel().getMutationSession();
		if (session != null) {
			MutationOperator[] operators = session.getOperators();
			fViewer.setInput(operators);
			setContentDescription("Total "+operators.length + " operators used");
		} else {
			setContentDescription("");
		}
	}
	
	protected void showSummary() {
		if (! CONTENT_SUMMARY.equals(fMemento.getString(SHOW_CONTENT))) {
			return;
		}
		
		fViewer.setInput(Activator.getMutationModel());
		
		MutationSession session = Activator.getMutationModel().getMutationSession();
		if (session != null) {
			int count = session.getMutationsCount();
			int alive = session.getMutationsAlive();
			setContentDescription(alive + " out of total "+ count + " responses are alive");
		} else {
			setContentDescription("");
		}
	}
	
	protected void openSelection(Object object) {
		String className = null;
		String methodName = null;
		int line = 0;
		
		if (object instanceof ITestSuiteElement) {
			ITestSuiteElement element = (ITestSuiteElement) object;
			className = element.getSuiteTypeName();
			
		}
		
		if (object instanceof ITestCaseElement) {
			ITestCaseElement element = (ITestCaseElement) object;
			className = element.getTestClassName();
			methodName = element.getTestMethodName();
		}
		
		if (object instanceof TestCaseAdapter) {
			TestCaseAdapter element = (TestCaseAdapter) object;
			className = element.getClassName();
			methodName = element.getMethodName();
		}
		
		if (object instanceof MutationResult) {
			object = ((MutationResult)object).getParent();
		}
		
		if (object instanceof MutationElement) {
			MutationElement result = (MutationElement) object;
			
			line = result.getLine();
			className = result.getClassName();
			methodName = result.getMethodName();
		}
		
		if (className == null) {
			return;
		}
		
		Action action = new OpenTestAction(this, className, methodName, line);
		if (action.isEnabled()) {
			action.run();
		}
	}
	
	protected TreeViewer getJUnitTreeViewer(boolean force) {
		// XXX this method is a hack for https://bugs.eclipse.org/bugs/show_bug.cgi?id=230122 
		IViewPart view = getSite().getPage().findView("org.eclipse.jdt.junit.ResultView");
		
		if ((view == null) && !force)
			return null;
		
		if (view == null) {
			try {
				view = getSite().getPage().showView("org.eclipse.jdt.junit.ResultView");
			} catch (PartInitException e) {
				Activator.log(e);
				return null;
			}
		}
		
		TestRunnerViewPart junitView = (TestRunnerViewPart) view;
		TestViewer viewer = (TestViewer) getField(junitView, "fTestViewer");
		
		if (viewer == null) {
			return null;
		}
		
		return (TreeViewer) getField(viewer, "fTreeViewer");
	}
	
	private static Object getField(Object object, String name) {
		try {
			Field field = object.getClass().getDeclaredField(name);
			field.setAccessible(true);
			return field.get(object);
		} catch (SecurityException e) {
			Activator.log(e);
		} catch (NoSuchFieldException e) {
			Activator.log(e);
		} catch (IllegalArgumentException e) {
			Activator.log(e);
		} catch (IllegalAccessException e) {
			Activator.log(e);
		}
		
		return null;
	}

	public void dispose() {
		super.dispose();
		
		writeSettings();
		
		TreeViewer junitViewer = getJUnitTreeViewer(false);
		if (junitViewer != null) {
			junitViewer.removeSelectionChangedListener(junitViewListener);
		}
		
		JUnitCore.removeTestRunListener(testRunListener);
	}

	public void setFocus() {
		// whenever focus is set to this view, make sure we're linked to JUnit view
		TreeViewer junitViewer = getJUnitTreeViewer(true);
		if (junitViewer != null) {
			junitViewer.addSelectionChangedListener(junitViewListener);
		}
	}

	public void registerInfoMessage(String message) {
		getViewSite().getActionBars().getStatusLineManager().setMessage(message);
	}

	protected void readSettings() {
		Preferences prefs = Activator.getDefault().getPluginPreferences();
		
		prefs.setDefault(SHOW_CONTENT, CONTENT_SUMMARY);
		prefs.setDefault(FILTER_DEAD, Boolean.FALSE.toString());
		
		fMemento.putString(SHOW_CONTENT, prefs.getString(SHOW_CONTENT));
		fMemento.putString(FILTER_DEAD, prefs.getString(FILTER_DEAD));
	}
	
	protected void writeSettings() {
		Preferences prefs = Activator.getDefault().getPluginPreferences();
		
		prefs.setValue(SHOW_CONTENT, fMemento.getString(SHOW_CONTENT));
		prefs.setValue(FILTER_DEAD, fMemento.getString(FILTER_DEAD));
	}
}
