package pl.poznan.put.qjunit.model;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.debug.core.ILaunch;
import org.eclipse.jdt.core.IJavaProject;

import pl.poznan.put.qjunit.Activator;

public class MutationSession {
	
	/**
	 * The server socket
	 */
	private ServerSocket fServerSocket;
	private Socket fSocket;
	private BufferedReader fBufferedReader;
	
	private IJavaProject fJavaProject;
	private ObjectDeserializer deserializer;
	
	private List fMutationElements = new ArrayList();
	private Map operators = new HashMap();
	
	/**
	 * Reads the message stream from the RemoteTestRunner
	 */
	private class ServerConnection extends Thread {
		int fServerPort;
		
		public ServerConnection(int port) {
			super("ServerConnection"); //$NON-NLS-1$
			fServerPort= port;
		}
		
		public void run() {
			try {
				fServerSocket= new ServerSocket(fServerPort);
				fSocket= fServerSocket.accept();	
				try {
				    fBufferedReader= new BufferedReader(new InputStreamReader(fSocket.getInputStream(), "UTF-8")); //$NON-NLS-1$
				} catch (UnsupportedEncodingException e) {
				    fBufferedReader= new BufferedReader(new InputStreamReader(fSocket.getInputStream()));				    
				}
				String message;
				while(fBufferedReader != null && (message= readMessage(fBufferedReader)) != null)
					receiveMessage(message);
			} catch (SocketException e) {
				//notifyTestRunTerminated();
			} catch (IOException e) {
				Activator.log(e);
				// fall through
			}
			shutDown();
		}
	}
	
	public MutationSession(ILaunch launch, IJavaProject javaProject, int port) {
		fJavaProject = javaProject;
		
		deserializer = new ObjectDeserializer();
		
		ServerConnection connection= new ServerConnection(port);
		connection.start();
	}
	
	private String readMessage(BufferedReader in) throws IOException {
		return in.readLine();
	}
	
	/**
	 * Read MutationInfo.toString()
	 * 
	 * "RESPONSE_TYPE;FILE_NAME;LINE;LEARNED;GENERATED;RESULTS;MESSAGES;TIMING" 
	 * @param message
	 */
	private void receiveMessage(String message) {
		MutationElement element;
		StringTokenizer st = new StringTokenizer(message, ";");
		String testClass = null;
		String testName = null;
		int responseType = -1;
		
		try {
			testClass = new String(st.nextToken());
			testName = new String(st.nextToken());
			
			element = new MutationElement(testClass, testName);
			responseType = Integer.parseInt(st.nextToken());
			element.setResponseType(responseType);
			element.setFileName(new String(st.nextToken()));
			element.setLine(Integer.parseInt(st.nextToken()));
			element.setSignature(new String(st.nextToken()));
			
			String[] learnedStrings = readArray(st);
			Object[] learned = new Object[learnedStrings.length];
			for (int i = 0; i < learnedStrings.length; i++) {
				learned[i] = deserializer.create("learned", learnedStrings[i], responseType);
			}
			element.setLearned(learned);
		} catch (RuntimeException e) {
			Activator.log("Error parsing message="+message, e);
			return;
		}
		
		try {
			String[] generators = readArray(st);
			String[] generated = readArray(st);
			String[] results = readArray(st);
			String[] resultMessages = readArray(st);
			long[] timings = readLongArray(st);
			
			
			MutationResult[] mutationResults = new MutationResult[generated.length];
			for (int i = 0; i < mutationResults.length; i++) {
				Object resp = deserializer.create("response", generated[i], responseType);
				mutationResults[i] = new MutationResult(element, generators[i], resp, results[i], resultMessages[i], timings[i]);
				updateOperator(generators[i], mutationResults[i]);
			}
			element.setResults(mutationResults);
		} catch (RuntimeException e) {
			Activator.log(e);
		}
		
		st = null;
		
		TestCaseAdapter tc = getTestCaseAdapter(testClass, testName);
		tc.put(element);
		
	}
	
	private void updateOperator(String operator, MutationResult mutationResult) {
		MutationOperator oper = (MutationOperator) operators.get(operator);
		if (oper == null) {
			oper = new MutationOperator(operator);
			operators.put(operator, oper);
		}
		
		oper.addResult(mutationResult);
	}
	
	private String[] readArray(StringTokenizer st) {
		int count = 0;
		try {
			count = Integer.parseInt(st.nextToken());
		} catch (NumberFormatException e) {
			Activator.log(e);
		};
		
		String[] result = new String[count];
		for (int i = 0; i < count; i++) {
			result[i] = new String(st.nextToken());
		}
		return result;
	}
	
	private long[] readLongArray(StringTokenizer st) {
		int count = 0;
		try {
			count = Integer.parseInt(st.nextToken());
		} catch (NumberFormatException e) {
			Activator.log(e);
		};
		
		long[] result = new long[count];
		for (int i = 0; i < count; i++) {
			result[i] = Long.parseLong(st.nextToken());
		}
		return result;
	}
	
	private synchronized void shutDown() {
		try {
			if (fBufferedReader != null) {
				fBufferedReader.close();
				fBufferedReader= null;
			}
		} catch(IOException e) {
		}	
		try {
			if (fSocket != null) {
				fSocket.close();
				fSocket= null;
			}
		} catch(IOException e) {
		}
		try{
			if (fServerSocket != null) {
				fServerSocket.close();
				fServerSocket= null;
			}
		} catch(IOException e) {
		}
	}
	
	public TestCaseAdapter getTestCaseAdapter(String className, String methodName) {
		TestCaseAdapter tmp = new TestCaseAdapter(className, methodName);
		int i = fMutationElements.indexOf(tmp);
		if (i == -1) {
			fMutationElements.add(tmp);
		} else {
			tmp = (TestCaseAdapter) fMutationElements.get(i);
		}
		
		return tmp;
	}
	
	public TestCaseAdapter[] getTestCases() {
		return (TestCaseAdapter[]) fMutationElements.toArray(new TestCaseAdapter[fMutationElements.size()]);
	}
	
	public MutationOperator[] getOperators() {
		Collection allOperators = operators.values();
		
		return (MutationOperator[]) allOperators.toArray(new MutationOperator[allOperators.size()]);
	}
	
	public int getMutationsCount() {
		int count = 0;
		for (Iterator i = fMutationElements.iterator(); i.hasNext(); ) {
			count += ((TestCaseAdapter) i.next()).getMutationsCount();
		}
		
		return count;
	}
	
	public int getMutationsAlive() {
		int count = 0;
		for (Iterator i = fMutationElements.iterator(); i.hasNext(); ) {
			count += ((TestCaseAdapter) i.next()).getMutationsAlive();
		}
		
		return count;
	}
	
	public IJavaProject getJavaProject() {
		return fJavaProject;
	}
}
