package pl.poznan.put.qjunit.model;

import java.util.HashSet;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchListener;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.internal.junit.launcher.JUnitLaunchConfigurationConstants;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import pl.poznan.put.qjunit.Activator;
import pl.poznan.put.qjunit.core.InstrumentationHelper;
import pl.poznan.put.qjunit.core.LaunchConfigurationDelegate;
import pl.poznan.put.qjunit.view.MutationsView;

/**
 * Stores mutations data from launches
 *
 */
public class MutationModel {
	
	private static class ResourceDeltaVisitor implements IResourceDeltaVisitor {

		public boolean visit(IResourceDelta delta) throws CoreException {
			IResource resource = delta.getResource();
			if (resource == null) {
				return true;
			}
			
			IProject project = resource.getProject();
			if (project == null)
				return true;
			
			InstrumentationHelper.removeInstrumentation(project.getName());
			
			return false;
		}
		
	}
	
	private static class ResourceChangeListener implements IResourceChangeListener {

		ResourceDeltaVisitor visitor = new ResourceDeltaVisitor();
		
		public void resourceChanged(IResourceChangeEvent event) {
			try {
				event.getDelta().accept(visitor);
			} catch (CoreException e) {
				Activator.log(e);
			}
		}

	}
	
	private class LaunchListener implements ILaunchListener {

		/**
		 * Used to track new launches. We need to do this
		 * so that we only attach a TestRunner once to a launch.
		 * Once a test runner is connected, it is removed from the set.
		 */
		private HashSet fTrackedLaunches= new HashSet(20);

		/*
		 * @see ILaunchListener#launchAdded(ILaunch)
		 */
		public void launchAdded(ILaunch launch) {
			fTrackedLaunches.add(launch);
		}

		/*
		 * @see ILaunchListener#launchRemoved(ILaunch)
		 */
		public void launchRemoved(final ILaunch launch) {
			fTrackedLaunches.remove(launch);
		}

		/*
		 * @see ILaunchListener#launchChanged(ILaunch)
		 */
		public void launchChanged(final ILaunch launch) {
			if (!fTrackedLaunches.contains(launch))
				return;
		
			ILaunchConfiguration config= launch.getLaunchConfiguration();
			if (config == null)
				return;
			
			final IJavaProject javaProject= JUnitLaunchConfigurationConstants.getJavaProject(config);
			if (javaProject == null)
				return;
			
			// test whether the launch defines the JUnit attributes
			String portStr= launch.getAttribute(LaunchConfigurationDelegate.ATTR_PORT);
			if (portStr == null)
				return;
			
			try {
				final int port= Integer.parseInt(portStr);
				fTrackedLaunches.remove(launch);
				getDisplay().asyncExec(new Runnable() {
					public void run() {
						connectTestRunner(launch, javaProject, port);
					}
				});
			} catch (NumberFormatException e) {
				return;
			}
		}

		private void connectTestRunner(ILaunch launch, IJavaProject javaProject, int port) {
			showMutationsView();
			
			fMutationSession = new MutationSession(launch, javaProject, port);
		}

		private MutationsView showMutationsView() {
			IWorkbenchWindow window = Activator.getDefault().getWorkbench().getActiveWorkbenchWindow();
			if (window == null)
				return null;
			
			IWorkbenchPage page= window.getActivePage();
			if (page == null)
				return null;
			
			try {
				return (MutationsView) page.showView(MutationsView.ID);
			} catch (PartInitException e) {
				Activator.log(e);
				return null;
			}
		}

		private Display getDisplay() {
			Display display= Display.getCurrent();
			if (display == null) {
				display= Display.getDefault();
			}
			return display;
		}
	}
	
	private ILaunchListener fLaunchListener;
	
	private MutationSession fMutationSession;
	
	private IResourceChangeListener fResourceChangeListener;
	
	public MutationModel() {
		fLaunchListener = new LaunchListener();
		fResourceChangeListener = new ResourceChangeListener();
	}
	
	public void start() {
		DebugPlugin.getDefault().getLaunchManager().addLaunchListener(fLaunchListener);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(fResourceChangeListener);
	}
	
	public void stop() {
		DebugPlugin.getDefault().getLaunchManager().removeLaunchListener(fLaunchListener);
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(fResourceChangeListener);
	}
	
	public MutationSession getMutationSession() {
		return fMutationSession;
	}
}
