package pl.poznan.put.qjunit.response.providers.operators;

import java.lang.reflect.Field;
import java.security.SecureClassLoader;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.generic.ClassGen;

/**
 * Utility methods for manipulating objects or classes.
 *
 */
public class OperatorsUtils {

	public static final String PROXY_NAME = "qjunitproxy.";
	
	/**
	 * Simple loader to load classes from byte[] array.
	 */
	protected static class DirectLoader extends SecureClassLoader
	{
	    protected DirectLoader() {
	        super(OperatorsUtils.class.getClassLoader());
	    }
	    
	    protected Class load(String name, byte[] data) {
	        return super.defineClass(name, data, 0, data.length);
	    }
	}
	
	public static DirectLoader loader = new DirectLoader(); 
	
	/**
	 * Tries to create a new instance of class.
	 * Note: This method may use some heuristics in future.
	 * 
	 * @param clazz
	 * @return new instance or null
	 */
	public static Object newInstance(Class clazz) {
		try {
			return clazz.newInstance();
		} catch (IllegalAccessException e) {
		} catch (InstantiationException e) {
		}
		
		return null;
	}
	
	/**
	 * Creates proxy for given instance of object of type clazz by creating
	 * type extending clazz.
	 * 
	 * @param clazz Type to override
	 * @param instance Instance of class
	 * @return proxy or null
	 */
	public static Object createProxy(Class clazz, Object instance) {
		String className = PROXY_NAME + clazz.getName();
		ClassGen cg = new ClassGen( className, clazz.getName(), "<generated>", Constants.ACC_PUBLIC | Constants.ACC_SUPER, null);
		cg.addEmptyConstructor(Constants.ACC_PUBLIC);
		
		byte[] bytes = cg.getJavaClass().getBytes();
		
		Class c = loader.load(className, bytes);
		
		try {
			return c.newInstance();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InstantiationException e) {
			e.printStackTrace();
		}
		
		return null;
	}
	
	/**
	 * Set primitive value of given field in object to given value.
	 * 
	 * @param field Field, which value should be set
	 * @param type Type of field
	 * @param obj Instance of object on which field value should be set
	 * @param value New value
	 * 
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 */
	static protected void setPrimitive(Field field, Class type, Object obj, Object value) throws IllegalArgumentException, IllegalAccessException {
		if (type.equals(Integer.TYPE)) {
			field.setInt(obj, ((Integer)value).intValue());
		} else if (type.equals(Boolean.TYPE)) {
			field.setBoolean(obj, ((Boolean)value).booleanValue());
		} else if (type.equals(Byte.TYPE)) {
			field.setByte(obj, ((Byte)value).byteValue());
		} else if (type.equals(Character.TYPE)) {
			field.setChar(obj, ((Character)value).charValue());
		} else if (type.equals(Double.TYPE)) {
			field.setDouble(obj, ((Double)value).doubleValue());
		} else if (type.equals(Float.TYPE)) {
			field.setFloat(obj, ((Float)value).floatValue());
		} else if (type.equals(Long.TYPE)) {
			field.setLong(obj, ((Long)value).longValue());
		} else if (type.equals(Short.TYPE)) {
			field.setShort(obj, ((Short)value).shortValue());
		}
	}
	
	/**
	 * Get value of primitive field as an object.
	 * 
	 * @param field Field containing the value
	 * @param type Type of field
	 * @param obj Object of which field value is to be returned
	 * @return Field value or null if not a primitive.
	 * 
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 */
	static protected Object getPrimitive(Field field, Class type, Object obj) throws IllegalArgumentException, IllegalAccessException {
		if (type.equals(Integer.TYPE)) {
			return Integer.valueOf(field.getInt(obj));
		} else if (type.equals(Boolean.TYPE)) {
			return Boolean.valueOf(field.getBoolean(obj));
		} else if (type.equals(Byte.TYPE)) {
			return Byte.valueOf(field.getByte(obj));
		} else if (type.equals(Character.TYPE)) {
			return Character.valueOf(field.getChar(obj));
		} else if (type.equals(Double.TYPE)) {
			return Double.valueOf(field.getDouble(obj));
		} else if (type.equals(Float.TYPE)) {
			return Float.valueOf(field.getFloat(obj));
		} else if (type.equals(Long.TYPE)) {
			return Long.valueOf(field.getLong(obj));
		} else if (type.equals(Short.TYPE)) {
			return Short.valueOf(field.getShort(obj));
		}
		return null;
	}
}
